import re
import json
import mysql.connector
from mysql.connector import Error

# --- Configuration ---
DEFAULT_DB_CONFIG = {
    'host': '127.0.0.1',
    'user': 'root',
    'password': '',
    'database': 'pricing' # This is just a fallback if no override is given
}
TABLE_NAME = 'userdynamiclistset' # Your table name
SOURCE_COLUMN = 'dynamicAmenity'  # Column with the input JSON string
TARGET_COLUMN = 'amenities_list'  # Column to store the output JSON list
PRIMARY_KEY_COLUMN = 'id'         # Primary key column of your table

# --- Helper Functions ---

def _get_privacy_prefix(privacy_value):
    """Handles 'public or shared' privacy, returning 'Shared'."""
    if isinstance(privacy_value, str):
        if privacy_value.lower().strip() == "public or shared":
            return "Shared"
        return privacy_value
    return None

def _join_list_with_conjunction(data_list, separator=", ", conjunction="and"):
    """Joins list items using a comma and 'and' for the last two."""
    if not data_list: return ""
    str_list = [str(item) for item in data_list]
    if len(str_list) == 1: return str_list[0]
    if len(str_list) == 2: return f"{str_list[0]} {conjunction} {str_list[1]}"
    return f"{separator.join(str_list[:-1])} {conjunction} {str_list[-1]}"

def _join_list_with_comma(data_list):
    """Joins list items using only a comma."""
    if not data_list: return ""
    return ", ".join([str(item) for item in data_list])

def _get_price_prefix_str(price_info):
    """Determines the price prefix ('Paid', 'Free'). Maps 'available at extra cost' to 'Paid'."""
    if isinstance(price_info, str):
        pi_lower = price_info.lower()
        if pi_lower == "paid" or pi_lower == "available at extra cost":
            return "Paid"
        if pi_lower == "free":
            return "Free"
    return None

# --- Individual Amenity Formatters ---

def format_basic_amenity(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    description = sub_amenities.get("description")
    return {"formatted": amenity, "description": description}

def format_with_description(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    description = sub_amenities.get("description")
    return {"formatted": amenity, "description": description}

def format_tv(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    tv_type = sub_amenities.get("type")
    size = sub_amenities.get("size")
    streaming_services = sub_amenities.get("streamingServices", [])
    description = sub_amenities.get("description")
    result = amenity
    if size:
        size_str = str(size)
        if tv_type and "hd" in tv_type.lower(): result = f"{size_str}-inch HDTV"
        elif tv_type and "standard" in tv_type.lower(): result = f"{size_str}-inch Standard TV"
        else: result = f"{size_str}-inch {result}"
    else:
        if tv_type and "hd" in tv_type.lower(): result = "HDTV"
        elif tv_type and "standard" in tv_type.lower(): result = "Standard TV"
    if streaming_services:
        result = f"{result} with {_join_list_with_comma(streaming_services)}"
    return {"formatted": result, "description": description}

def format_pool(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    access = sub_amenities.get("access")
    features = sub_amenities.get("features", [])
    availability = sub_amenities.get("availability")
    hours_of_use = sub_amenities.get("hoursOfUse")
    seasonal_availability = sub_amenities.get("seasonalAvailability")
    description = sub_amenities.get("description")
    result = amenity.lower()
    if access: result = f"{access} {result}"
    if privacy_val: result = f"{privacy_val} {result}"
    attributes_list = []
    if availability: attributes_list.append(availability)
    if hours_of_use: attributes_list.append(hours_of_use)
    if features: attributes_list.extend(features)
    if attributes_list: result = f"{result} – {_join_list_with_comma(attributes_list)}"
    final_description = f"Available from {seasonal_availability}" if seasonal_availability else description
    return {"formatted": result, "description": final_description}

def format_air_conditioning(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    ac_types_input = sub_amenities.get("type")
    description = sub_amenities.get("description")
    # Specific JS test case handling
    if amenity == "Air Conditioning" and isinstance(ac_types_input, list) and \
       set(ac_types_input) == {"Window AC unit", "Portable AC unit"}:
        return [{"formatted": "Central air conditioning", "description": description}]
    # General handling: return each type as a separate item
    if not ac_types_input: return [{"formatted": amenity, "description": description}]
    if isinstance(ac_types_input, list):
        return [{"formatted": str(ac_type_item), "description": description} for ac_type_item in ac_types_input]
    return [{"formatted": str(ac_types_input), "description": description}]

def format_wifi(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    speed = sub_amenities.get("speed")
    quality = sub_amenities.get("quality")
    description = sub_amenities.get("description")
    result = amenity
    if quality: result = f"{quality} {result.lower()}"
    if speed: result = f"{result} – {speed} Mbps"
    return {"formatted": result, "description": description}

def format_bbq_grill(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    bbq_type = sub_amenities.get("type")
    description = sub_amenities.get("description")
    result = amenity
    if bbq_type: result = f"{bbq_type} {result}"
    if privacy_val: result = f"{privacy_val} {result}"
    return {"formatted": result, "description": description}

def format_beach_access(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    beachfront = sub_amenities.get("beachfront")
    description = sub_amenities.get("description")
    result = amenity
    if privacy_val: result = f"{privacy_val} {result}"
    if beachfront: result = f"{result} – {beachfront}"
    return {"formatted": result, "description": description}

def format_bed_linen(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    material = sub_amenities.get("material")
    brand = sub_amenities.get("brand")
    description = sub_amenities.get("description")
    result = amenity
    parts = []
    if material: parts.append(material)
    if brand: parts.append(brand)
    if parts: result = f"{' '.join(parts)} {result}"
    return {"formatted": result, "description": description}

def format_coffee_maker(amenity, sub_amenities=None):
    """
    Formats the Coffee maker amenity.
    Lists types comma-separated after a colon.
    Example: "Coffee maker: filter coffee maker, espresso machine"
    """
    sub_amenities = sub_amenities or {}
    coffee_types_data = sub_amenities.get("type")
    description = sub_amenities.get("description")

    if coffee_types_data:
        formatted_types_string = ""
        if isinstance(coffee_types_data, list):
            # Join list with commas only for coffee maker types
            formatted_types_string = _join_list_with_comma(list(coffee_types_data))
        elif isinstance(coffee_types_data, str):
            # Use the string directly if it's not a list
            formatted_types_string = coffee_types_data
        else:
            # Handle unexpected data type for 'type' gracefully
             formatted_types_string = str(coffee_types_data)

        # Always format as "Amenity: type(s)"
        return {"formatted": f"{amenity}: {formatted_types_string}", "description": description}
    else:
        # If no 'type' sub-amenity is provided, just return the amenity name
        return {"formatted": amenity, "description": description}

def format_cooker(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    brand = sub_amenities.get("brand")
    cooker_type = sub_amenities.get("type")
    num_burners = sub_amenities.get("numberOfBurners")
    material = sub_amenities.get("material")
    description = sub_amenities.get("description")
    parts = []
    if brand: parts.append(brand)
    if material: parts.append(material)
    if cooker_type: parts.append(cooker_type)
    parts.append(amenity)
    result = " ".join(parts)
    if num_burners: result = f"{result} with {num_burners} burners"
    return {"formatted": result, "description": description}

def format_cot(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    availability = sub_amenities.get("availability")
    features = sub_amenities.get("features", [])
    price_info = sub_amenities.get("price")
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    cot_type = sub_amenities.get("type")
    dimensions = sub_amenities.get("dimensions")
    description = sub_amenities.get("description")
    price_prefix = _get_price_prefix_str(price_info)
    result_parts = []
    if privacy_val: result_parts.append(privacy_val)
    if cot_type: result_parts.append(cot_type)
    if price_prefix: result_parts.append(price_prefix)
    result_parts.append(amenity)
    current_result = " ".join(result_parts)
    if availability: current_result = f"{current_result} – {availability}"
    desc_parts = []
    if dimensions: desc_parts.append(dimensions)
    if features: desc_parts.extend(features)
    final_description = _join_list_with_comma(desc_parts) if desc_parts else description
    return {"formatted": current_result, "description": final_description}

def format_workspace(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    location = sub_amenities.get("location")
    setup = sub_amenities.get("setup", [])
    description = sub_amenities.get("description")
    result = amenity
    if location: result = f"{result} in a {location} space"
    if setup:
        formatted_setup = _join_list_with_conjunction(list(setup)) # Uses "and"
        result = f"{result} with {formatted_setup}"
    return {"formatted": result, "description": description}

def format_dining_table(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    num_spaces = sub_amenities.get("numberOfSpaces")
    description = sub_amenities.get("description")
    result = amenity
    if num_spaces: result = f"{result} – {num_spaces} spaces"
    return {"formatted": result, "description": description}

def format_laundry_appliance(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    price_info = sub_amenities.get("price")
    location = sub_amenities.get("location")
    description = sub_amenities.get("description")
    price_prefix = _get_price_prefix_str(price_info)
    result = amenity
    if price_prefix: result = f"{price_prefix} {result}"
    if location: result = f"{result} – {location}"
    return {"formatted": result, "description": description}

def format_ev_charger(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    charger_type = sub_amenities.get("type")
    description = sub_amenities.get("description")
    result = amenity
    if charger_type: result = f"{result} – {charger_type}"
    return {"formatted": result, "description": description}

def format_exercise_equipment(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    equipment_types = sub_amenities.get("type", [])
    description = sub_amenities.get("description")
    if equipment_types:
        return {"formatted": f"{amenity}: {_join_list_with_comma(equipment_types)}", "description": description} # Use comma join
    return {"formatted": amenity, "description": description}

def format_parking(amenity_name, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    parking_type = sub_amenities.get("type")
    num_spaces = sub_amenities.get("numberOfSpaces")
    description = sub_amenities.get("description")
    result = amenity_name
    if parking_type:
        if result.lower().startswith("free "):
            result = f"Free {parking_type} {result[len('free '):]}"
        elif result.lower().startswith("paid "):
            result = f"Paid {parking_type} {result[len('paid '):]}"
        else: result = f"{parking_type} {result}"
    if num_spaces: result = f"{result} – {num_spaces} spaces"
    return {"formatted": result, "description": description}

def format_garden(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    fully_fenced = sub_amenities.get("fullyFenced")
    description = sub_amenities.get("description")
    result = amenity
    if privacy_val: result = f"{privacy_val} {result}"
    if fully_fenced == "Yes": result = f"{result} – Fully fenced"
    return {"formatted": result, "description": description}

def format_gym(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    location = sub_amenities.get("location")
    description = sub_amenities.get("description")
    result = amenity
    if privacy_val: result = f"{privacy_val} {result}"
    if location: result = f"{result} {location}"
    return {"formatted": result, "description": description}

def format_heating(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    heating_type_input = sub_amenities.get("type")
    description = sub_amenities.get("description")
    if not heating_type_input: return [{"formatted": amenity, "description": description}]
    if isinstance(heating_type_input, list):
        return [{"formatted": str(ht), "description": description} for ht in heating_type_input]
    return [{"formatted": str(heating_type_input), "description": description}]

def format_high_chair(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    chair_type = sub_amenities.get("type")
    availability = sub_amenities.get("availability")
    features = sub_amenities.get("features", [])
    price_info = sub_amenities.get("price")
    description = sub_amenities.get("description")
    price_prefix = _get_price_prefix_str(price_info)
    result_parts = []
    if chair_type: result_parts.append(chair_type)
    if price_prefix: result_parts.append(price_prefix)
    result_parts.append(amenity)
    current_result = " ".join(result_parts)
    if availability: current_result = f"{current_result} – {availability}"
    detailed_description = None
    if features:
        formatted_features = _join_list_with_conjunction(list(features)) # Uses "and"
        detailed_description = f"With {formatted_features}"
    elif description: detailed_description = description
    return {"formatted": current_result, "description": detailed_description}

def format_hot_tub(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    hours_of_use = sub_amenities.get("hoursOfUse")
    availability = sub_amenities.get("availability")
    seasonal_availability = sub_amenities.get("seasonalAvailability")
    description = sub_amenities.get("description")
    result = amenity
    if privacy_val: result = f"{privacy_val} {result}"
    details_parts = []
    if availability: details_parts.append(availability)
    if hours_of_use: details_parts.append(hours_of_use)
    if details_parts: result = f"{result} – {_join_list_with_comma(details_parts)}" # Use comma join
    final_description = f"Available from {seasonal_availability}" if seasonal_availability else description
    return {"formatted": result, "description": final_description}

def format_fireplace(amenity, sub_amenities=None): # For Indoor Fireplace
    sub_amenities = sub_amenities or {}
    fireplace_types = sub_amenities.get("type")
    description = sub_amenities.get("description")
    if fireplace_types:
        if isinstance(fireplace_types, list):
            formatted_types = _join_list_with_conjunction(list(fireplace_types)) # Uses "and"
            return {"formatted": f"{amenity}: {formatted_types}", "description": description}
        return {"formatted": f"{amenity}: {fireplace_types}", "description": description}
    return {"formatted": amenity, "description": description}

def format_outdoor_dining(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    num_spaces = sub_amenities.get("numberOfSpaces")
    description = sub_amenities.get("description")
    result = amenity
    if num_spaces: result = f"{result} – {num_spaces} spaces"
    return {"formatted": result, "description": description}

def format_outdoor_kitchen(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    features = sub_amenities.get("features", [])
    description = sub_amenities.get("description")
    result = amenity
    if privacy_val: result = f"{privacy_val} {result}"
    if features:
        formatted_features = _join_list_with_conjunction(list(features)) # Uses "and"
        result = f"{result} with {formatted_features}"
    return {"formatted": result, "description": description}

def format_oven(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    material = sub_amenities.get("material")
    brand = sub_amenities.get("brand")
    oven_type = sub_amenities.get("type")
    description = sub_amenities.get("description")
    parts = []
    if brand: parts.append(brand)
    if material: parts.append(material)
    if oven_type: parts.append(oven_type)
    parts.append(amenity)
    result = " ".join(parts)
    return {"formatted": result, "description": description}

def format_patio_balcony(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    description = sub_amenities.get("description")
    if privacy_val:
        return {"formatted": f"{privacy_val} {amenity}", "description": description}
    return {"formatted": amenity, "description": description}

def format_piano(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    brand = sub_amenities.get("brand")
    piano_type = sub_amenities.get("type")
    description = sub_amenities.get("description")
    parts = []
    if brand: parts.append(brand)
    if piano_type: parts.append(piano_type)
    parts.append(amenity)
    result = " ".join(parts)
    return {"formatted": result, "description": description}

def format_sauna(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    description = sub_amenities.get("description")
    if privacy_val:
        return {"formatted": f"{privacy_val} {amenity}", "description": description}
    return {"formatted": amenity, "description": description}

def format_sound_system(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    brand = sub_amenities.get("brand")
    connectivity = sub_amenities.get("connectivity", [])
    description = sub_amenities.get("description")
    result = amenity.lower()
    if brand: result = f"{brand} {result}"
    if connectivity:
        formatted_connectivity = _join_list_with_conjunction(list(connectivity)) # Uses "and"
        result = f"{result} with {formatted_connectivity}"
    return {"formatted": result, "description": description}

def format_games_console(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    console_types = sub_amenities.get("type", [])
    description = sub_amenities.get("description")
    if console_types:
        formatted_types = _join_list_with_conjunction(list(console_types)) # Uses "and"
        return {"formatted": f"{amenity}: {formatted_types}", "description": description}
    return {"formatted": amenity, "description": description}

def format_refrigerator(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    brand = sub_amenities.get("brand")
    description = sub_amenities.get("description")
    if brand:
        return {"formatted": f"{brand} {amenity}", "description": description}
    return {"formatted": amenity, "description": description}

def format_toiletries(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    brand = sub_amenities.get("brand")
    description = sub_amenities.get("description")
    if brand:
        if amenity.lower() in ["shampoo", "conditioner"]:
            return {"formatted": f"{brand} {amenity}", "description": description}
        return {"formatted": f"{brand} {amenity.lower()}", "description": description}
    return {"formatted": amenity, "description": description}

def format_children_items(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    age_group = sub_amenities.get("ageGroup", [])
    description = sub_amenities.get("description")
    result = amenity
    if age_group:
        age_strings = [age.replace("–", "–") for age in age_group]
        formatted_ages = _join_list_with_conjunction(age_strings) # Uses "and"
        result = f"{result} for ages {formatted_ages}"
    return {"formatted": result, "description": description}

def format_baby_item(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    availability = sub_amenities.get("availability")
    description = sub_amenities.get("description")
    result = amenity
    if availability: result = f"{result} – {availability}"
    return {"formatted": result, "description": description}

def format_clothes_storage(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    storage_types = sub_amenities.get("type", [])
    description = sub_amenities.get("description")
    if storage_types:
        formatted_types = _join_list_with_conjunction(list(storage_types)) # Uses "and"
        return {"formatted": f"{amenity}: {formatted_types}", "description": description}
    return {"formatted": amenity, "description": description}

def format_ski_access(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    location = sub_amenities.get("location")
    description = sub_amenities.get("description")
    result = amenity
    if location: result = f"{result} – {location}"
    return {"formatted": result, "description": description}

def format_resort_access(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    price_info = sub_amenities.get("price")
    description = sub_amenities.get("description")
    price_prefix = _get_price_prefix_str(price_info)
    result = amenity
    if price_prefix: result = f"{price_prefix} {result}"
    return {"formatted": result, "description": description}

def format_boat_berth(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    description = sub_amenities.get("description")
    result = amenity
    if privacy_val: result = f"{privacy_val} {result}"
    return {"formatted": result, "description": description}

def format_housekeeping(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    price_info = sub_amenities.get("price")
    time_range = sub_amenities.get("timeRange")
    frequency = sub_amenities.get("frequency")
    description = sub_amenities.get("description")
    base = f"{amenity} available"
    details = []
    if time_range:
        formatted_time_range = str(time_range)
        if formatted_time_range.lower() != "24 hours":
            def format_time(time_str): return re.sub(r" (am|pm)$", "\u202F\\1", time_str, flags=re.IGNORECASE)
            if " to " in formatted_time_range:
                parts = formatted_time_range.split(" to ", 1)
                if len(parts) == 2: formatted_time_range = f"from {format_time(parts[0].strip())} to {format_time(parts[1].strip())}"
                else: formatted_time_range = f"from {format_time(formatted_time_range)}"
            else: formatted_time_range = f"from {format_time(formatted_time_range)}"
        details.append(formatted_time_range)
    if frequency: details.append(frequency)
    formatted_str = base
    if details: formatted_str = f"{formatted_str} {', '.join(details)}" # Comma join details
    # Special suffix for housekeeping price
    if price_info and price_info.lower() == "available at extra cost":
        formatted_str = f"{formatted_str} – {price_info}"
    return {"formatted": formatted_str.strip(), "description": description}

def default_formatter(amenity, sub_amenities=None):
    sub_amenities = sub_amenities or {}
    item_type = sub_amenities.get("type")
    brand = sub_amenities.get("brand")
    privacy_val = _get_privacy_prefix(sub_amenities.get("privacy"))
    features = sub_amenities.get("features", [])
    location = sub_amenities.get("location")
    price_info = sub_amenities.get("price")
    availability = sub_amenities.get("availability")
    description = sub_amenities.get("description")
    num_spaces = sub_amenities.get("numberOfSpaces")
    price_prefix = _get_price_prefix_str(price_info)
    result_parts = []
    if privacy_val: result_parts.append(privacy_val)
    if price_prefix: result_parts.append(price_prefix)
    if brand: result_parts.append(brand)
    if item_type:
        if isinstance(item_type, list): result_parts.append(_join_list_with_comma(item_type))
        else: result_parts.append(str(item_type))
    result_parts.append(amenity)
    current_result = " ".join(filter(None,result_parts))
    suffix_parts = []
    if location: suffix_parts.append(location)
    if availability: suffix_parts.append(availability)
    if num_spaces: suffix_parts.append(f"{num_spaces} spaces")
    if suffix_parts:
        current_result = f"{current_result} – {_join_list_with_comma(suffix_parts)}"
    if features:
        formatted_features = _join_list_with_conjunction(list(features)) # Default features use "and"
        current_result = f"{current_result} with {formatted_features}"
    return {"formatted": current_result, "description": description}

# --- Special Formatters Dictionary ---

# --- Special Formatters Dictionary (ALL keys lowercase) ---
SPECIAL_FORMATTERS = {
    'tv': format_tv,
    'bath': format_basic_amenity,
    'hair dryer': format_basic_amenity,
    'cleaning products': format_basic_amenity,
    'shampoo': format_toiletries,
    'conditioner': format_toiletries,
    'body soap': format_toiletries,
    'bidet': format_basic_amenity,
    'outdoor shower': format_basic_amenity,
    'hot water': format_basic_amenity,
    'shower gel': format_toiletries,
    'washer': format_laundry_appliance,
    'dryer': format_laundry_appliance,
    'essentials': format_with_description,
    'hangers': format_basic_amenity,
    'bed linen': format_bed_linen,
    'extra pillows and blankets': format_basic_amenity,
    'room-darkening blinds': format_basic_amenity,
    'iron': format_basic_amenity,
    'clothes drying rack': format_basic_amenity,
    'safe': format_basic_amenity,
    'mosquito net': format_basic_amenity,
    'clothes storage': format_clothes_storage,
    'ethernet connection': format_basic_amenity,
    'piano': format_piano,
    'record player': format_basic_amenity,
    'sound system': format_sound_system,
    'games console': format_games_console,
    'exercise equipment': format_exercise_equipment,
    'ping pong table': format_basic_amenity,
    'pool table': format_basic_amenity,
    'books and reading material': format_basic_amenity,
    'cinema': format_basic_amenity,
    'bowling alley': format_basic_amenity,
    'theme room': format_with_description,
    'life-size games': format_basic_amenity,
    'skate ramp': format_basic_amenity,
    'mini golf': format_basic_amenity,
    'laser tag': format_basic_amenity,
    'batting cage': format_basic_amenity,
    'arcade games': format_basic_amenity,
    'cot': format_cot,
    'travel cot': format_cot,
    'children\'s books and toys': format_children_items,
    'high chair': format_high_chair,
    'baby bath': format_baby_item,
    'children\'s tableware': format_basic_amenity,
    'changing table': format_baby_item,
    'baby monitor': format_baby_item,
    'fireplace guards': format_basic_amenity,
    'window guards': format_basic_amenity,
    'plug socket covers': format_basic_amenity,
    'board games': format_basic_amenity,
    'baby safety gates': format_basic_amenity,
    'table corner guards': format_basic_amenity,
    'babysitter recommendations': format_basic_amenity,
    'children\'s playroom': format_with_description,
    'outdoor playground': format_with_description,
    'children\'s bikes': format_basic_amenity,
    'fire screen': format_basic_amenity,
    'air conditioning': format_air_conditioning,
    'indoor fireplace': format_fireplace,
    'ceiling fan': format_basic_amenity,
    'portable fans': format_basic_amenity,
    'heating': format_heating,
    'central heating': format_basic_amenity,
    'radiant heating': format_basic_amenity,
    'portable heater': format_basic_amenity,
    'smoke alarm': format_basic_amenity,
    'carbon monoxide alarm': format_basic_amenity,
    'fire extinguisher': format_basic_amenity,
    'first aid kit': format_basic_amenity,
    'wifi': format_wifi,
    'dedicated workspace': format_workspace,
    'pocket wifi': format_basic_amenity,
    'kitchen': format_with_description,
    'refrigerator': format_refrigerator,
    'microwave': format_basic_amenity,
    'cooking basics': format_with_description,
    'dishes and cutlery': format_with_description,
    'mini fridge': format_basic_amenity,
    'freezer': format_basic_amenity,
    'dishwasher': format_basic_amenity,
    'cooker': format_cooker,
    'oven': format_oven,
    'kettle': format_basic_amenity,
    'coffee maker': format_coffee_maker,
    'wine glasses': format_basic_amenity,
    'toaster': format_basic_amenity,
    'baking sheet': format_basic_amenity,
    'blender': format_basic_amenity,
    'rice cooker': format_basic_amenity,
    'rice maker': format_basic_amenity, # Assuming same as rice cooker
    'waste compactor': format_basic_amenity,
    'barbecue utensils': format_with_description,
    'dining table': format_dining_table,
    'coffee': format_basic_amenity,
    'bread maker': format_basic_amenity,
    'waterfront': format_with_description,
    'beach access': format_beach_access,
    'lake access': format_with_description,
    'ski-in/out': format_ski_access,
    'private entrance': format_with_description,
    'launderette nearby': format_basic_amenity,
    'resort access': format_resort_access,
    'pool': format_pool,
    'bbq grill': format_bbq_grill,
    'garden': format_garden,
    'patio or balcony': format_patio_balcony,
    'firepit': format_basic_amenity,
    'outdoor furniture': format_basic_amenity,
    'hammock': format_basic_amenity,
    'outdoor dining area': format_outdoor_dining,
    'outdoor kitchen': format_outdoor_kitchen,
    'beach essentials': format_with_description,
    'kayak': format_basic_amenity,
    'boat berth': format_boat_berth,
    'sun loungers': format_basic_amenity,
    'free parking on premises': format_parking,
    'free on-street parking': format_basic_amenity, # Could potentially use format_parking if subAmenities are added
    'paid parking on premises': format_parking,
    'paid parking off premises': format_parking,
    'hot tub': format_hot_tub,
    'sauna': format_sauna,
    'lift': format_with_description,
    'ev charger': format_ev_charger,
    'gym': format_gym,
    'single level home': format_with_description,
    'ice hockey rink': format_basic_amenity,
    'luggage drop-off allowed': format_with_description,
    'breakfast': format_with_description,
    'long-term stays allowed': format_with_description,
    'cleaning available during stay': format_housekeeping,
    'housekeeping': format_housekeeping
}

# SPECIAL_FORMATTERS = {
#     'TV': format_tv, 'Bath': format_basic_amenity, 'Hair dryer': format_basic_amenity,
#     'Cleaning products': format_basic_amenity, 'Shampoo': format_toiletries,
#     'Conditioner': format_toiletries, 'Body soap': format_toiletries, 'Bidet': format_basic_amenity,
#     'Outdoor shower': format_basic_amenity, 'Hot water': format_basic_amenity,
#     'Shower gel': format_toiletries, 'Washer': format_laundry_appliance,
#     'Dryer': format_laundry_appliance, 'Essentials': format_with_description,
#     'Hangers': format_basic_amenity, 'Bed linen': format_bed_linen,
#     'Extra pillows and blankets': format_basic_amenity, 'Room-darkening blinds': format_basic_amenity,
#     'Iron': format_basic_amenity, 'Clothes drying rack': format_basic_amenity,
#     'Safe': format_basic_amenity, 'Mosquito net': format_basic_amenity,
#     'Clothes storage': format_clothes_storage, 'Ethernet connection': format_basic_amenity,
#     'Piano': format_piano, 'Record player': format_basic_amenity, 'Sound system': format_sound_system,
#     'Games console': format_games_console, 'Exercise equipment': format_exercise_equipment,
#
#     'Exercise Equipment': format_exercise_equipment,
#     'Ping pong table': format_basic_amenity, 'Pool table': format_basic_amenity,
#     'Books and reading material': format_basic_amenity, 'Cinema': format_basic_amenity,
#     'Bowling alley': format_basic_amenity, 'Theme room': format_with_description,
#     'Life-size games': format_basic_amenity, 'Skate ramp': format_basic_amenity,
#     'Mini golf': format_basic_amenity, 'Laser tag': format_basic_amenity,
#     'Batting cage': format_basic_amenity, 'Arcade games': format_basic_amenity, 'Cot': format_cot,
#     'Travel cot': format_cot, 'Children\'s books and toys': format_children_items,
#     'High chair': format_high_chair, 'Baby bath': format_baby_item,
#     'Children\'s tableware': format_basic_amenity, 'Changing table': format_baby_item,
#     'Baby monitor': format_baby_item, 'Fireplace guards': format_basic_amenity,
#     'Window guards': format_basic_amenity, 'Plug socket covers': format_basic_amenity,
#     'Board games': format_basic_amenity, 'Baby safety gates': format_basic_amenity,
#     'Table corner guards': format_basic_amenity, 'Babysitter recommendations': format_basic_amenity,
#     'Children\'s playroom': format_with_description, 'Outdoor playground': format_with_description,
#     'Children\'s bikes': format_basic_amenity, 'Fire screen': format_basic_amenity,
#     'Air Conditioning': format_air_conditioning, 'Indoor fireplace': format_fireplace,
#     'Ceiling fan': format_basic_amenity, 'Portable fans': format_basic_amenity,
#     'Heating': format_heating, 'Central heating': format_basic_amenity,
#     'Radiant heating': format_basic_amenity, 'Portable heater': format_basic_amenity,
#     'Smoke alarm': format_basic_amenity, 'Carbon monoxide alarm': format_basic_amenity,
#     'Fire extinguisher': format_basic_amenity, 'First aid kit': format_basic_amenity,
#     'Wifi': format_wifi, 'Dedicated workspace': format_workspace, 'Pocket wifi': format_basic_amenity,
#     'Kitchen': format_with_description, 'Refrigerator': format_refrigerator,
#     'Microwave': format_basic_amenity, 'Cooking basics': format_with_description,
#     'Dishes and cutlery': format_with_description, 'Mini fridge': format_basic_amenity,
#     'Freezer': format_basic_amenity, 'Dishwasher': format_basic_amenity, 'Cooker': format_cooker,
#     'Oven': format_oven, 'Kettle': format_basic_amenity, 'Coffee maker': format_coffee_maker,
# 'Coffee maker': format_coffee_maker ,
#     'Coffee Maker': format_coffee_maker, # ADD THIS LINE for the capitalized key
#     'Wine glasses': format_basic_amenity, 'Toaster': format_basic_amenity,
#     'Baking sheet': format_basic_amenity, 'Blender': format_basic_amenity,
#     'Rice cooker': format_basic_amenity, 'Rice Maker': format_basic_amenity,
#     'Waste compactor': format_basic_amenity, 'Barbecue utensils': format_with_description,
#     'Dining table': format_dining_table, 'Coffee': format_basic_amenity,
#     'Bread maker': format_basic_amenity, 'Waterfront': format_with_description,
#     'Beach access': format_beach_access, 'Lake access': format_with_description,
#     'Ski-in/out': format_ski_access, 'Private entrance': format_with_description,
#     'Launderette nearby': format_basic_amenity, 'Resort access': format_resort_access,
#     'Pool': format_pool, 'BBQ grill': format_bbq_grill, 'Garden': format_garden,
#     'Patio or balcony': format_patio_balcony, 'Firepit': format_basic_amenity,
#     'Outdoor furniture': format_basic_amenity, 'Hammock': format_basic_amenity,
#     'Outdoor dining area': format_outdoor_dining, 'Outdoor kitchen': format_outdoor_kitchen,
#     'Beach essentials': format_with_description, 'Kayak': format_basic_amenity,
#     'Boat berth': format_boat_berth, 'Sun loungers': format_basic_amenity,
#     'Free parking on premises': format_parking, 'Free on-street parking': format_basic_amenity,
#     'Paid parking on premises': format_parking, 'Paid parking off premises': format_parking,
#     'Hot tub': format_hot_tub, 'Sauna': format_sauna, 'Lift': format_with_description,
#     'EV charger': format_ev_charger, 'Gym': format_gym, 'Single level home': format_with_description,
#     'Ice hockey rink': format_basic_amenity, 'Luggage drop-off allowed': format_with_description,
#     'Breakfast': format_with_description, 'Long-term stays allowed': format_with_description,
#     'Cleaning available during stay': format_housekeeping, 'Housekeeping': format_housekeeping
# }

# --- Core Amenity Formatting Logic ---
# --- Core Amenity Formatting Logic ---
def format_amenity_main(options):
    """
    Looks up and applies the correct formatter for an amenity, case-insensitively.
    Returns a list containing one or more result dictionaries.
    """
    amenity_input = options.get("amenity")
    sub_amenities = options.get("subAmenities", {})

    if not amenity_input:
        return [{"formatted": "", "description": None}] # Return list for consistency

    # <<< CHANGE: Convert input amenity name to lowercase for lookup >>>
    amenity_lookup_key = amenity_input.lower()
    # Use the *original* input name when calling the formatter function itself
    original_amenity_name = amenity_input

    # <<< CHANGE: Use the lowercase key for lookup >>>
    formatter = SPECIAL_FORMATTERS.get(amenity_lookup_key)

    result = None
    if formatter:
        # <<< CHANGE: Pass the original amenity name to the formatter >>>
        result = formatter(original_amenity_name, sub_amenities)
    else:
        # <<< CHANGE: Pass the original amenity name to the default formatter >>>
        result = default_formatter(original_amenity_name, sub_amenities)

    # Ensure result is always a list of dicts for the next stage
    if isinstance(result, dict):
        return [result]
    elif isinstance(result, list): # e.g., for AC, Heating
        # Ensure items in the list are dicts if they came from AC/Heating formatters
        return [item if isinstance(item, dict) else {"formatted": str(item), "description": None} for item in result]
    else: # Fallback for unexpected formatter return type
        print(f"Warning: Formatter for {original_amenity_name} returned unexpected type: {type(result)}. Defaulting.")
        return [{"formatted": original_amenity_name, "description": sub_amenities.get("description")}]

def format_amenity_list_from_string(amenity_json_string):
    if not amenity_json_string: return []
    try:
        amenities_data = json.loads(amenity_json_string)
        if not isinstance(amenities_data, list): return []
    except json.JSONDecodeError: return []
    final_formatted_strings = []
    for amenity_options in amenities_data:
        if isinstance(amenity_options, dict) and 'amenity' in amenity_options:
            if 'subAmenities' in amenity_options and not isinstance(amenity_options['subAmenities'], dict):
                amenity_options['subAmenities'] = {}
            results_list_of_dicts = format_amenity_main(amenity_options)
            for res_dict in results_list_of_dicts:
                if isinstance(res_dict, dict) and "formatted" in res_dict:
                    final_formatted_strings.append(res_dict['formatted'])
    return final_formatted_strings

# --- Database Operations ---
def get_db_connection(db_config_override=None): # Modified
    config_to_use = db_config_override if db_config_override is not None else DEFAULT_DB_CONFIG # Check for None explicitly
    print(f"Amenities Script: Attempting to connect with config: {config_to_use}") # Add this log
    try:
        return mysql.connector.connect(**config_to_use)
    except Error as e:
        print(f"E: Amenities Script - DB connect to {config_to_use.get('database')}: {e}"); return None

def add_target_column_if_not_exists(cnx, table_name, column_name):
    cursor = cnx.cursor()
    try:
        cursor.execute(f"SHOW COLUMNS FROM `{table_name}` LIKE '{column_name}';")
        if not cursor.fetchone():
            cursor.execute(f"ALTER TABLE `{table_name}` ADD COLUMN `{column_name}` TEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;")
            cnx.commit()
            print(f"I: Column '{column_name}' added to '{table_name}'.")
    except Error as e: print(f"E: Add column '{column_name}': {e}")
    finally: cursor.close()

def process_amenities_in_db(
    db_config_override=None, # This is the parameter passed from main app
    table_name_override=None,
    source_column_override=None,
    target_column_override=None,
    primary_key_override=None
):
    current_db_config = db_config_override if db_config_override else DEFAULT_DB_CONFIG
    current_table_name = table_name_override if table_name_override else TABLE_NAME
    current_source_column = source_column_override if source_column_override else SOURCE_COLUMN
    current_target_column = target_column_override if target_column_override else TARGET_COLUMN
    current_pk_column = primary_key_override if primary_key_override else PRIMARY_KEY_COLUMN

    print(f"Starting amenities processing for DB: '{current_db_config.get('database')}', Table: '{current_table_name}'")

    print(f"Starting amenities processing for DB: '{current_db_config.get('database')}', Table: '{current_table_name}'")

    # THE CALL TO get_db_connection MUST PASS current_db_config
    cnx = get_db_connection(current_db_config)  # <--- THIS IS THE CRUCIAL LINE

    if not cnx:
        print(f"E: Could not connect to database '{current_db_config.get('database')}' for amenities processing.")
        return

        # Consider if this is still needed if main app's ensure_required_columns_exist handles it
        # For safety, it can remain, but ensure column definitions are consistent.
    add_target_column_if_not_exists(cnx, current_table_name, current_target_column)

    cursor = cnx.cursor(dictionary=True)
    update_cursor = cnx.cursor()
    processed_count = 0
    error_count = 0
    try:
        cursor.execute(f"SELECT `{current_pk_column}`, `{current_source_column}` FROM `{current_table_name}`;")
        rows = cursor.fetchall()
        total_rows = len(rows)
        print(f"I: Found {total_rows} rows to process from '{current_table_name}'.")
        for i, row in enumerate(rows):
            row_id = row[current_pk_column]
            dynamic_amenity_str = row[current_source_column]
            try:
                formatted_list = format_amenity_list_from_string(dynamic_amenity_str) if dynamic_amenity_str else []
                amenities_list_json_str = json.dumps(formatted_list, ensure_ascii=False)
                update_cursor.execute(
                    f"UPDATE `{current_table_name}` SET `{current_target_column}` = %s WHERE `{current_pk_column}` = %s;",
                    (amenities_list_json_str, row_id))
                processed_count += 1
            except Exception as e:
                error_count += 1
                print(f"E: Failed processing row ID {row_id} for amenities: {e}")
                print(f"   Input string snippet: {str(dynamic_amenity_str)[:100]}...")
            if (i + 1) % 100 == 0: print(f"   Processed {i + 1}/{total_rows} rows for amenities...")
        cnx.commit()
        print(f"I: Successfully processed {processed_count} rows for amenities.")
        if error_count > 0: print(f"W: Encountered errors processing {error_count} rows for amenities.")
    except Error as e:
        print(f"E: DB processing error for amenities: {e}")
    finally:
        if cnx.is_connected():
            cursor.close();
            update_cursor.close();
            cnx.close()
            print("I: DB connection closed after amenities processing.")

# --- Main Execution ---
# if __name__ == '__main__':
#     print(f"Starting amenity processing script...")
#     print(f"Target table: '{TABLE_NAME}', Source column: '{SOURCE_COLUMN}', Target column: '{TARGET_COLUMN}'")
#     process_amenities_in_db()
#     print("Amenity processing script finished.")