import json
from typing import Dict, Optional, Any # Added Any
from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
import pandas as pd
from contextlib import asynccontextmanager
import asyncio
import re
import psutil
import os # Added os for file path handling

from pydantic import BaseModel
from typing import Dict, List, Any, Optional
from fastapi import FastAPI, HTTPException
from fastapi.middleware.cors import CORSMiddleware
import pandas as pd
from contextlib import asynccontextmanager
import asyncio
import re
import psutil
from typing_extensions import Literal
import mysql.connector
from mysql.connector import Error
from datetime import datetime, timedelta
import random

# --- Existing Imports and Functions ---
# (Keep all your existing imports like get_top_amenities, generate_recommendations, etc.)
from top_10_amenities import get_top_amenities
from min_stay_rec import generate_recommendations
from base_price import get_recommended_price
from filter200 import process_dataframe

# --- Constants ---
SETTINGS_FILE = "customization_settings.json"

# --- Pydantic Models ---

# Model for the unified pricing endpoint (Keep this if you use it elsewhere)
class UnifiedPricing(BaseModel):
    pricing_type: Literal["custom", "algorithm", "market"]
    base_price: float
    min_price: float
    max_price: float
    market_segment: Optional[str] = None

# Model for the customization settings payload
class CustomizationSettings(BaseModel):
    occupancyEnabled: bool
    occupancyProfile: str # e.g., "default", "aggressive", "custom"
    customPricingData: Dict[str, Dict[str, Any]] # Allow Any type for values initially (string or number)
    minStayEnabled: bool
    minStayProfile: str # e.g., "default", "custom"
    customMinStayData: Dict[str, Dict[str, Any]] # Allow Any type for values initially

# --- Global Variables ---
processed_data: Dict[str, Optional[Any]] = { # Changed Optional[dict] to Optional[Any] for flexibility
    "map_recs": None,
    "min_stay_rec": None,
    "amenities": None,
    "pricing": None,
    "market_pricing": None,
    "monthly_averages": None,
    "applied_pricing": None, # Keep this if used
    # Customization settings will be loaded from file, not stored here initially
}
data_processed = False

# --- Helper Functions ---

def print_system_usage():
    # (Keep your existing function)
    cpu_usage = psutil.cpu_percent(interval=1)
    mem_info = psutil.virtual_memory()
    ram_used = mem_info.used
    ram_total = mem_info.total
    ram_percent = mem_info.percent
    gb = 1024 ** 3
    ram_used_gb = ram_used / gb
    ram_total_gb = ram_total / gb
    print(f"CPU Usage: {cpu_usage}%")
    print(f"RAM Usage: {ram_used_gb:.2f} GB used out of {ram_total_gb:.2f} GB total ({ram_percent}%)")

def generate_market_pricing(df):
    # (Keep your existing function)
    price_percentiles = {
        "top_10": df['price'].quantile(0.90),
        "top_25": df['price'].quantile(0.75),
        "top_50": df['price'].quantile(0.50),
        "top_65": df['price'].quantile(0.35)
    }
    pricing_categories = {
        "luxury": {"base_price": round(float(price_percentiles["top_10"]), 2), "max_price": round(float(price_percentiles["top_10"] * 2), 2), "min_price": round(float(price_percentiles["top_10"] * 0.4), 2)},
        "upscale": {"base_price": round(float(price_percentiles["top_25"]), 2), "max_price": round(float(price_percentiles["top_25"] * 2), 2), "min_price": round(float(price_percentiles["top_25"] * 0.4), 2)},
        "midscale": {"base_price": round(float(price_percentiles["top_50"]), 2), "max_price": round(float(price_percentiles["top_50"] * 2), 2), "min_price": round(float(price_percentiles["top_50"] * 0.4), 2)},
        "economy": {"base_price": round(float(price_percentiles["top_65"]), 2), "max_price": round(float(price_percentiles["top_65"] * 2), 2), "min_price": round(float(price_percentiles["top_65"] * 0.4), 2)}
    }
    return pricing_categories

def timestamp_to_str(obj):
    # (Keep your existing function)
    if hasattr(obj, 'strftime'):
        return obj.strftime('%Y-%m-%d')
    elif isinstance(obj, list):
        return [timestamp_to_str(item) for item in obj]
    elif isinstance(obj, dict):
        return {k: timestamp_to_str(v) for k, v in obj.items()}
    else:
        return obj

def calculate_monthly_averages(prop_df, market_df):
    # (Keep your existing function)
    pattern = r'^[A-Z][a-z]{2}_\d{4}$'
    def process_df(df):
        monthly_data = {}
        for col in df.columns:
            if re.match(pattern, col):
                try:
                    avg = round(df[col].mean(), 2)
                    monthly_data[col] = float(avg) if not pd.isna(avg) else None
                except Exception as e:
                    print(f"Error processing column {col}: {e}")
                    monthly_data[col] = None
        return monthly_data
    return {"property": process_df(prop_df), "market": process_df(market_df)}

# --- Lifespan Function ---
@asynccontextmanager
async def lifespan(app: FastAPI):
    # (Keep your existing lifespan logic)
    asyncio.create_task(process_data())
    yield
    pass

# --- FastAPI App Setup ---
app = FastAPI(lifespan=lifespan)

app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"], # Adjust for production
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# --- Data Processing Function ---
async def process_data():
    # (Keep your existing process_data function)
    # This function populates processed_data["min_stay_rec"] which is needed
    global data_processed, processed_data
    try:
        print("Starting data processing...")
        prop_1 = pd.read_csv("processed_test.csv")
        connection = None # Initialize connection to None
        try:
            connection = mysql.connector.connect(
                host='127.0.0.1',
                user='paperbirdtech_admin',
                password='2IkTxtwNSqsR',
                database='paperbirdtech_ptbeta'
            )
            if connection.is_connected():
                print("Connected to MySQL for rental_properties data")
                general = pd.read_sql_query("SELECT * FROM rental_properties", connection)
            else:
                raise Exception("Failed to connect to MySQL")
        except Error as e:
            print(f"Error connecting to MySQL: {e}")
            # Decide how to handle this: raise, exit, or use fallback data?
            # For now, let's re-raise to indicate a critical failure
            raise e
        finally:
            if connection and connection.is_connected():
                connection.close()
                print("MySQL connection closed")

        general_prop1_200 = process_dataframe(general, lat=42.1713, long=-73.9698)

        # --- Process all data types ---
        tasks = []

        async def process_amenities():
            try:
                processed_data["amenities"] = get_top_amenities(general, prop_1)
                print("Amenities data processed.")
            except Exception as e: print(f"Error processing amenities: {e}"); processed_data["amenities"] = {"error": str(e)}

        async def process_market_pricing():
            try:
                processed_data["market_pricing"] = generate_market_pricing(general_prop1_200)
                print("Market pricing tiers processed.")
            except Exception as e: print(f"Error processing market pricing: {e}"); processed_data["market_pricing"] = {"error": str(e)}

        async def process_min_stay():
            try:
                min_stay_rec_data = generate_recommendations(general_prop1_200, prop_1)
                processed_data["min_stay_rec"] = timestamp_to_str(min_stay_rec_data)
                print("Min stay recommendations processed.")
            except Exception as e: print(f"Error processing min stay: {e}"); processed_data["min_stay_rec"] = {"error": str(e)}

        async def process_pricing():
            try:
                base_price = get_recommended_price(general_prop1_200, prop_1)
                processed_data["pricing"] = {"base_price": float(base_price), "max_price": float(base_price * 2), "min_price": float(base_price * 0.4)}
                print("Pricing data processed.")
            except Exception as e: print(f"Error processing pricing: {e}"); processed_data["pricing"] = {"error": str(e)}

        async def process_monthly_averages():
            try:
                processed_data["monthly_averages"] = calculate_monthly_averages(prop_1, general_prop1_200)
                print("Monthly averages processed.")
            except Exception as e: print(f"Error processing monthly averages: {e}"); processed_data["monthly_averages"] = {"error": str(e)}

        async def process_map_recs():
            try:
                selected_columns = general_prop1_200[['price', 'lat', 'long', 'color']].copy()
                processed_data["map_recs"] = selected_columns.to_dict(orient='records')
                print("Map recommendations processed.")
            except Exception as e: print(f"Error processing map data: {e}"); processed_data["map_recs"] = {"error": str(e)}

        # Run processing tasks concurrently
        await asyncio.gather(
            process_amenities(),
            process_market_pricing(),
            process_min_stay(),
            process_pricing(),
            process_monthly_averages(),
            process_map_recs()
        )

        data_processed = True
        print("All data processed and ready.")
        print_system_usage() # Optional: Check usage after processing

    except Exception as e:
        print(f"Critical error during data processing setup: {e}")
        data_processed = False # Ensure flag is false if setup fails


# --- Helper Functions for Customization ---

def load_customization_settings() -> Dict:
    """Loads customization settings from the JSON file."""
    if not os.path.exists(SETTINGS_FILE):
        # Return default settings if file doesn't exist
        return {
            "occupancyEnabled": True,
            "occupancyProfile": "default",
            "customPricingData": {},
            "minStayEnabled": True,
            "minStayProfile": "default",
            "customMinStayData": {},
        }
    try:
        with open(SETTINGS_FILE, 'r') as f:
            return json.load(f)
    except (json.JSONDecodeError, IOError) as e:
        print(f"Error loading settings file ({SETTINGS_FILE}): {e}. Returning defaults.")
        # Return defaults in case of error reading the file
        return {
            "occupancyEnabled": True,
            "occupancyProfile": "default",
            "customPricingData": {},
            "minStayEnabled": True,
            "minStayProfile": "default",
            "customMinStayData": {},
        }

def save_customization_settings(settings: Dict):
    """Saves customization settings to the JSON file."""
    try:
        with open(SETTINGS_FILE, 'w') as f:
            json.dump(settings, f, indent=4)
        print("Customization settings saved successfully.")
    except IOError as e:
        print(f"Error saving settings file ({SETTINGS_FILE}): {e}")
        raise HTTPException(status_code=500, detail="Failed to save customization settings.")


# --- API Endpoints ---

# GET endpoints for processed data (Keep these as they are)
@app.get("/api/map-recs")
async def get_map_recs():
    if not data_processed: return {"status": "processing"}
    if processed_data["map_recs"] is None or "error" in processed_data["map_recs"]: raise HTTPException(500, "Map data processing failed or not available")
    return processed_data["map_recs"]

@app.get("/api/min-stay-rec")
async def get_min_stay_rec():
    # This endpoint provides the *default* recommendations
    if not data_processed: return {"status": "processing"}
    if processed_data["min_stay_rec"] is None or ("error" in processed_data["min_stay_rec"]):
         # Check if it's the specific error dictionary or just None
         error_detail = "Minimum stay data processing failed or not available"
         if isinstance(processed_data["min_stay_rec"], dict) and "error" in processed_data["min_stay_rec"]:
             error_detail = f"Minimum stay data processing error: {processed_data['min_stay_rec']['error']}"
         raise HTTPException(status_code=500, detail=error_detail)
    return processed_data["min_stay_rec"]

@app.get("/api/amenities")
async def get_amenities():
    if not data_processed: return {"status": "processing"}
    if processed_data["amenities"] is None or "error" in processed_data["amenities"]: raise HTTPException(500, "Amenities data processing failed or not available")
    return processed_data["amenities"]

@app.get("/api/pricing")
async def get_pricing():
    if not data_processed: return {"status": "processing"}
    if processed_data["pricing"] is None or "error" in processed_data["pricing"]: raise HTTPException(500, "Pricing data processing failed or not available")
    return processed_data["pricing"]

@app.get("/api/monthly-averages")
async def get_monthly_averages():
    if not data_processed: return {"status": "processing"}
    if processed_data["monthly_averages"] is None or "error" in processed_data["monthly_averages"]: raise HTTPException(500, "Monthly averages processing failed or not available")
    return processed_data["monthly_averages"]

@app.get("/api/market-pricing")
async def get_market_pricing():
    if not data_processed: return {"status": "processing"}
    if processed_data["market_pricing"] is None or "error" in processed_data["market_pricing"]: raise HTTPException(500, "Market pricing data processing failed or not available")
    return processed_data["market_pricing"]

@app.get("/api/status")
async def get_status():
    return {"processed": data_processed}

@app.post("/api/refresh")
async def refresh_data():
    global data_processed
    data_processed = False # Mark as processing
    asyncio.create_task(process_data())
    return {"status": "Data refresh initiated"}

@app.get("/api/system-usage")
async def get_system_usage():
    # (Keep your existing endpoint)
    try:
        cpu_usage = psutil.cpu_percent(interval=1)
        virtual_mem = psutil.virtual_memory()
        memory_usage = {"total": virtual_mem.total, "available": virtual_mem.available, "used": virtual_mem.used, "percent": virtual_mem.percent}
        process = psutil.Process()
        process_cpu = process.cpu_percent(interval=None) # Use interval=None for non-blocking call after first call
        process_memory = process.memory_info()
        process_stats = {"cpu_percent": process_cpu, "memory_rss": process_memory.rss, "memory_vms": process_memory.vms}
        return {"system": {"cpu_usage": cpu_usage, "memory_usage": memory_usage}, "process": process_stats}
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Error retrieving system usage: {str(e)}")

# --- NEW Customization Endpoints ---

@app.get("/api/customization")
async def get_customization_settings():
    """Endpoint to retrieve the current customization settings."""
    settings = load_customization_settings()
    return settings

@app.post("/api/customization")
async def update_customization_settings(settings: CustomizationSettings):
    """Endpoint to save the customization settings."""
    try:
        # Pydantic already validated the structure based on CustomizationSettings model
        save_customization_settings(settings.dict()) # Convert Pydantic model to dict for saving
        return {"message": "Customization settings saved successfully."}
    except HTTPException as he:
        # Re-raise HTTP exceptions from save_customization_settings
        raise he
    except Exception as e:
        # Catch any other unexpected errors during saving
        print(f"Unexpected error saving customization settings: {e}")
        raise HTTPException(status_code=500, detail=f"An unexpected error occurred: {str(e)}")


# --- Unified Pricing Endpoints (Keep if needed) ---

@app.post("/api/apply-pricing")
async def apply_pricing(pricing: UnifiedPricing):
    # (Keep your existing endpoint if you use it)
    global processed_data
    try:
        if pricing.pricing_type == "market" and not pricing.market_segment:
            raise HTTPException(400, "Market segment must be specified for market pricing")
        if "applied_pricing" not in processed_data: processed_data["applied_pricing"] = {}
        processed_data["applied_pricing"] = {
            "type": pricing.pricing_type, "base_price": pricing.base_price, "min_price": pricing.min_price,
            "max_price": pricing.max_price, "market_segment": pricing.market_segment if pricing.pricing_type == "market" else None
        }
        segment_info = f" ({pricing.market_segment})" if pricing.pricing_type == "market" else ""
        print(f"{pricing.pricing_type.capitalize()}{segment_info} pricing applied: Base=${pricing.base_price}, Min=${pricing.min_price}, Max=${pricing.max_price}")
        return {"status": "success", "message": f"{pricing.pricing_type.capitalize()}{segment_info} pricing applied successfully"}
    except HTTPException as he: raise he
    except Exception as e: raise HTTPException(500, f"Error applying pricing: {str(e)}")

@app.get("/api/applied-pricing")
async def get_applied_pricing():
    # (Keep your existing endpoint if you use it)
    if "applied_pricing" in processed_data and processed_data["applied_pricing"]:
        return processed_data["applied_pricing"]
    else:
        if data_processed and processed_data.get("pricing") and "error" not in processed_data["pricing"]:
             # Fallback to algorithm pricing if available and no error
             algo_pricing = processed_data["pricing"]
             return {"type": "algorithm", "base_price": algo_pricing["base_price"], "min_price": algo_pricing["min_price"], "max_price": algo_pricing["max_price"], "market_segment": None}
        else:
             # If algorithm pricing also failed or data isn't processed, indicate no pricing is set
             raise HTTPException(status_code=404, detail="No pricing has been applied yet or default pricing is unavailable.")


import json
import random
from datetime import datetime, timedelta
from typing import Dict, List, Any, Optional
from fastapi import FastAPI, HTTPException
from pydantic import BaseModel


# Add this Pydantic model for the date range request
class DateRangePricingRequest(BaseModel):
    start_date: str  # Format: YYYY-MM-DD
    end_date: str  # Format: YYYY-MM-DD


# Add this function to generate dates between start and end
def generate_date_range(start_date_str: str, end_date_str: str) -> List[str]:
    """Generate a list of dates between start_date and end_date (inclusive)."""
    try:
        start_date = datetime.strptime(start_date_str, "%Y-%m-%d")
        end_date = datetime.strptime(end_date_str, "%Y-%m-%d")

        if start_date > end_date:
            raise ValueError("Start date must be before or equal to end date")

        dates = []
        current_date = start_date
        while current_date <= end_date:
            dates.append(current_date.strftime("%Y-%m-%d"))
            current_date += timedelta(days=1)

        return dates
    except ValueError as e:
        raise ValueError(f"Invalid date format or range: {str(e)}")


# Add this function to apply random variation to price
def apply_price_variation(base_price: float, variation_percent: float = 5.0) -> float:
    """Apply a random variation within ±variation_percent to the base price."""
    variation_factor = 1 + (random.uniform(-variation_percent, variation_percent) / 100)
    varied_price = base_price * variation_factor
    return round(varied_price, 2)  # Round to 2 decimal places


# Add this function to fetch date-specific data
async def fetch_date_specific_data(start_date: str, end_date: str) -> pd.DataFrame:
    """Fetch data from MySQL for a specific table named in the format 'startdate-enddate-year'."""
    connection = None
    try:
        # Parse start and end dates
        start_dt = datetime.strptime(start_date, "%Y-%m-%d")
        end_dt = datetime.strptime(end_date, "%Y-%m-%d")

        # Format the table name in the required pattern: 21mar-23mar-2025
        start_day = start_dt.day
        end_day = end_dt.day
        month = start_dt.strftime("%b").lower()  # Get month abbreviation (mar)
        year = start_dt.year

        # Construct table name in the format '21mar-23mar-2025'
        table_name = f"{start_day}{month}-{end_day}{month}-{year}"

        connection = mysql.connector.connect(
            host='127.0.0.1',
            user='paperbirdtech_admin',
            password='2IkTxtwNSqsR',
            database='paperbirdtech_ptbeta'
        )
        if connection.is_connected():
            print(f"Connected to MySQL for fetching data from table '{table_name}'")

            # Query from the specific date range table
            query = f"SELECT * FROM `{table_name}`"
            try:
                date_specific_data = pd.read_sql_query(query, connection)
                print(f"Successfully fetched {len(date_specific_data)} records from table '{table_name}'")
                return date_specific_data
            except mysql.connector.Error as err:
                if err.errno == 1146:  # Table doesn't exist error
                    print(f"Table '{table_name}' does not exist")
                    # If the specific table doesn't exist, we have no fallback in this scenario
                    raise Exception(f"Required table '{table_name}' does not exist")
                else:
                    raise
        else:
            raise Exception("Failed to connect to MySQL")
    except Error as e:
        print(f"Error connecting to MySQL: {e}")
        raise e
    except Exception as e:
        print(f"Error fetching data from table '{table_name}': {e}")
        raise e
    finally:
        if connection and connection.is_connected():
            connection.close()
            print("MySQL connection closed")


# Add this new endpoint
# Add this GET endpoint for date range pricing that accepts query parameters
@app.get("/api/date-range-pricing")
async def get_date_range_pricing_get(start_date: str, end_date: str):
    """GET endpoint for date range pricing that accepts query parameters."""
    # Create a request object to reuse the POST endpoint logic
    request = DateRangePricingRequest(start_date=start_date, end_date=end_date)

    # Call the existing POST handler with this request
    return await get_date_range_pricing_post(request)


# Rename the POST endpoint function for clarity
@app.post("/api/date-range-pricing")
async def get_date_range_pricing_post(request: DateRangePricingRequest):
    """Calculate pricing for each day in the specified date range with variations."""
    if not data_processed:
        return {"status": "processing", "message": "Initial data processing not complete"}

    try:
        # Generate list of dates in the range
        dates = generate_date_range(request.start_date, request.end_date)

        # Fetch data from the table with the specific name format (21mar-23mar-2025)
        try:
            # Get data from the specialized table
            date_specific_data = await fetch_date_specific_data(request.start_date, request.end_date)

            # Process the data to get properties within the filtering radius (top 200)
            date_specific_filtered = process_dataframe(date_specific_data, lat=42.1713, long=-73.9698)

            # Read the test property data
            prop_1 = pd.read_csv("processed_test.csv")

            # Calculate recommended price based on the date-specific data
            date_specific_base_price = get_recommended_price(date_specific_filtered, prop_1)
            print(f"Date-specific base price calculated: ${date_specific_base_price}")

        except Exception as e:
            print(f"Error processing date-specific data: {e}")
            # No fallback in this case - we specifically need this table's data
            raise HTTPException(status_code=404,
                                detail=f"Could not retrieve or process data for the requested date range: {str(e)}")

        # Check if we have applied pricing
        try:
            applied_pricing_response = await get_applied_pricing()
            if isinstance(applied_pricing_response, dict) and "type" in applied_pricing_response:
                applied_pricing = applied_pricing_response
                print(f"Found applied pricing: {applied_pricing}")

                # Compare with algorithm pricing
                if processed_data["pricing"] and applied_pricing["base_price"] == processed_data["pricing"][
                    "base_price"]:
                    print("Applied pricing matches algorithm pricing - using date-specific price directly")
                    base_price_to_use = date_specific_base_price
                else:
                    # Calculate price difference and adjust applied pricing
                    general_base_price = processed_data["pricing"]["base_price"]
                    price_difference = date_specific_base_price - general_base_price
                    base_price_to_use = applied_pricing["base_price"] + price_difference
                    print(f"Applied pricing differs from algorithm pricing - adjusting by {price_difference}")
                    print(f"Original applied price: {applied_pricing['base_price']}, Adjusted: {base_price_to_use}")
            else:
                # No applied pricing, use date-specific price directly
                base_price_to_use = date_specific_base_price
                print(f"No applied pricing found - using date-specific price: {base_price_to_use}")
        except HTTPException:
            # No applied pricing found, use date-specific price directly
            base_price_to_use = date_specific_base_price
            print(f"No applied pricing accessible - using date-specific price: {base_price_to_use}")
        except Exception as e:
            print(f"Error checking applied pricing: {e}")
            # In case of error, default to date-specific pricing
            base_price_to_use = date_specific_base_price

        # Generate prices for each date with random ±5% variations
        result_prices = {}
        for date in dates:
            result_prices[date] = apply_price_variation(base_price_to_use)

        # Return the results
        return {
            "date_range": {
                "start": request.start_date,
                "end": request.end_date,
                "days": len(dates)
            },
            "base_price": float(base_price_to_use),  # Convert to float for JSON serialization
            "daily_prices": result_prices
        }

    except HTTPException as he:
        # Re-raise HTTP exceptions
        raise he
    except ValueError as ve:
        raise HTTPException(status_code=400, detail=str(ve))
    except Exception as e:
        print(f"Error generating date range pricing: {e}")
        raise HTTPException(status_code=500, detail=f"Error processing date range pricing: {str(e)}")

        # Check if we have applied pricing
        applied_pricing = None
        try:
            if "applied_pricing" in processed_data and processed_data["applied_pricing"]:
                applied_pricing = processed_data["applied_pricing"]
                print(f"Found applied pricing: {applied_pricing}")
        except Exception as e:
            print(f"Error checking applied pricing: {e}")

        # Calculate the pricing based on conditions
        result_prices = {}

        # Determine base price to use
        if applied_pricing:
            # If applied pricing exists, use that as the base with adjustment
            applied_base_price = applied_pricing["base_price"]

            # Calculate difference between date-specific and general pricing
            if date_specific_base_price is not None:
                general_base_price = processed_data["pricing"]["base_price"]
                price_difference = date_specific_base_price - general_base_price
                adjusted_base_price = applied_base_price + price_difference
                print(
                    f"Adjusted applied base price by diff ({price_difference}): {applied_base_price} → {adjusted_base_price}")
            else:
                # If date-specific pricing failed, just use applied pricing
                adjusted_base_price = applied_base_price
                print(f"Using applied base price with no adjustment: {adjusted_base_price}")
        else:
            # If no applied pricing, use the date-specific pricing or fall back to regular pricing
            if date_specific_base_price is not None:
                adjusted_base_price = date_specific_base_price
                print(f"Using date-specific base price: {adjusted_base_price}")
            else:
                # Fall back to the default pricing
                adjusted_base_price = processed_data["pricing"]["base_price"]
                print(f"Using default base price: {adjusted_base_price}")

        # Generate prices for each date with variations
        for date in dates:
            result_prices[date] = apply_price_variation(adjusted_base_price)

        # Return the results
        return {
            "date_range": {
                "start": request.start_date,
                "end": request.end_date,
                "days": len(dates)
            },
            "base_price": float(adjusted_base_price),  # Convert to float for JSON serialization
            "daily_prices": result_prices
        }

    except ValueError as ve:
        raise HTTPException(status_code=400, detail=str(ve))
    except Exception as e:
        print(f"Error generating date range pricing: {e}")
        raise HTTPException(status_code=500, detail=f"Error processing date range pricing: {str(e)}")




# --- Root Endpoint ---
@app.get("/")
async def root():
    return {"status": "Server is running", "data_processed": data_processed}

# --- Main Execution ---
if __name__ == "__main__":
    import uvicorn
    print("Starting FastAPI server...")
    # Ensure the settings file exists with defaults if it's missing before starting
    load_customization_settings()
    uvicorn.run(app, host="training.paperbirdtech.com", port=8080)