import json
from typing import Dict, Optional, Any, List, Set
from fastapi import FastAPI, HTTPException, Query, Request, Body
from fastapi.middleware.cors import CORSMiddleware
import pandas as pd
from contextlib import asynccontextmanager
import asyncio
import functools
import re
import psutil
from pydantic import BaseModel, Field, root_validator
from typing_extensions import Literal
import mysql.connector
from mysql.connector import Error as MySQLError
from datetime import datetime, timedelta, date
import random
import ast
from calendar import monthrange, SATURDAY, SUNDAY
import traceback

# --- SQL-based and other existing function imports ---
try:
    from sql_top_10_amenity import get_top_amenities
    from sql_base_price import get_recommended_price
    from sql_min_stay_rec import generate_recommendations
    from filter200 import select_comparables_for_pricing
    import sql_update_all_amenities
except ImportError as e:
    print(f"Warning: Failed to import one or more SQL/processing modules: {e}")


    def get_top_amenities(*args, **kwargs):
        return {"error": "Module get_top_amenities not found"}


    def get_recommended_price(*args, **kwargs):
        return None  # Important for logic relying on this


    def generate_recommendations(*args, **kwargs):
        return {"error": "Module generate_recommendations not found"}


    def select_comparables_for_pricing(df, *args, **kwargs):
        return df


    class sql_update_all_amenities:
        @staticmethod
        def process_amenities_in_db(*args, **kwargs): print("Warning: sql_update_all_amenities dummy called")

# --- Base DB Configuration ---
BASE_DB_CONFIG = {
    'host': '127.0.0.1',
    'user': 'paperbirdtech_admin',
    'password': '2IkTxtwNSqsR'
}
CENTRAL_DB_NAME = "paperbirdtech_ptbeta"
USER_LISTING_TABLE_NAME = "userdynamiclistsets"

# --- Constants for Customization & Processing ---
OCCUPANCY_PROFILE_WINDOWS = {
    "0_15_days": (0, 15), "16_30_days": (16, 30),
    "30_60_days": (31, 60), "60_90_days": (61, 90)
}
ALL_PREDEFINED_PRICING_PROFILES = {
    "default": {"0_15_days": {"≤10": -15, "≤20": -15, "≤30": -10, "≤40": -5, "≤50": -5, "≤60": 0, "≤70": 0, "≤80": 0,
                              "≤100": 0},
                "16_30_days": {"≤10": -10, "≤20": -10, "≤30": -5, "≤40": -5, "≤50": 0, "≤60": 0, "≤70": 0, "≤80": 5,
                               "≤100": 10},
                "30_60_days": {"≤10": -5, "≤20": -5, "≤30": -5, "≤40": 0, "≤50": 0, "≤60": 0, "≤70": 5, "≤80": 10,
                               "≤100": 15},
                "60_90_days": {"≤10": -5, "≤20": -5, "≤30": 0, "≤40": 0, "≤50": 0, "≤60": 5, "≤70": 10, "≤80": 15,
                               "≤100": 20}},
    "aggressive": {
        "0_15_days": {"≤10": -30, "≤20": -25, "≤30": -20, "≤40": -20, "≤50": -5, "≤60": 10, "≤70": 5, "≤80": 5,
                      "≤100": 0},
        "16_30_days": {"≤10": -20, "≤20": -20, "≤30": -15, "≤40": -10, "≤50": -10, "≤60": -5, "≤70": 0, "≤80": 0,
                       "≤100": 0},
        "30_60_days": {"≤10": -15, "≤20": -10, "≤30": -10, "≤40": -5, "≤50": 0, "≤60": 0, "≤70": 0, "≤80": 0,
                       "≤100": 5},
        "60_90_days": {"≤10": -10, "≤20": -5, "≤30": -5, "≤40": -5, "≤50": 0, "≤60": 0, "≤70": 0, "≤80": 5, "≤100": 5}},
    "step last minute": {
        "0_15_days": {"≤10": -35, "≤20": -35, "≤30": -30, "≤40": -25, "≤50": -20, "≤60": -15, "≤70": -10, "≤80": -5,
                      "≤100": 0},
        "16_30_days": {"≤10": -30, "≤20": -25, "≤30": -20, "≤40": -15, "≤50": -10, "≤60": -5, "≤70": 0, "≤80": 5,
                       "≤100": 5},
        "30_60_days": {"≤10": -25, "≤20": -20, "≤30": -15, "≤40": -10, "≤50": -5, "≤60": 0, "≤70": 0, "≤80": 0,
                       "≤100": 0},
        "60_90_days": {"≤10": 0, "≤20": 0, "≤30": 0, "≤40": 0, "≤50": 0, "≤60": 0, "≤70": 0, "≤80": 0, "≤100": 0}}
}

REQUIRED_COLUMNS_SCHEMA = {
    "occupancyEnabled": "BOOLEAN DEFAULT TRUE",
    "occupancyProfile": "VARCHAR(50) DEFAULT 'default'",
    "occupancyProfileDetails": "TEXT",
    "minStayEnabled": "BOOLEAN DEFAULT TRUE",
    "minStayProfile": "VARCHAR(50) DEFAULT 'default'",
    "customMinStayData": "TEXT",
    "daily_occupancy_adjustments_json": "TEXT",
    "active_pricing_profile": "VARCHAR(50) DEFAULT 'algorithmic'",
    "active_pricing_details": "TEXT DEFAULT NULL",
    "processed_amenities_json": "TEXT",
    "processed_pricing_json": "TEXT",
    "processed_min_stay_json": "TEXT",
    "processed_map_recs_json": "TEXT",
    "processed_monthly_averages_json": "TEXT",
    "processed_market_pricing_json": "TEXT",
    "calendar_pricing_json": "TEXT DEFAULT NULL",
    "specific_date_pricing_json": "TEXT DEFAULT NULL",  # <<< NEW COLUMN
    "processing_status": "VARCHAR(20) DEFAULT NULL",
    "last_processed_at": "DATETIME DEFAULT NULL"
}

USER_TABLE_CREATE_SQL = f"""
CREATE TABLE IF NOT EXISTS `{USER_LISTING_TABLE_NAME}` (
    listId VARCHAR(255) PRIMARY KEY,
    lat DECIMAL(10, 8) NULL,
    `long` DECIMAL(11, 8) NULL,
    available_dates TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX(lat, `long`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
"""

MARKET_TABLE_CREATE_SQL = """
CREATE TABLE IF NOT EXISTS `{table_name}` (
    id INT AUTO_INCREMENT PRIMARY KEY,
    listId VARCHAR(255) NULL UNIQUE,
    lat DECIMAL(10, 8) NULL,
    `long` DECIMAL(11, 8) NULL,
    price DECIMAL(10, 2) NULL,
    scraped_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX(lat, `long`),
    INDEX(price)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
"""


# --- Pydantic Models ---
class ListingProcessRequest(BaseModel):
    listing_id: str = Field(..., description="The unique identifier for the listing.")
    location_name: str = Field(..., description="The name of the location.")


class CustomizationUpdateRequest(BaseModel):
    listing_id: str
    location_name: str
    occupancyEnabled: bool
    occupancyProfile: str
    customPricingData: Dict[str, Dict[str, Any]] = Field(default_factory=dict)
    minStayEnabled: bool
    minStayProfile: str
    customMinStayData: Dict[str, Dict[str, Any]] = Field(default_factory=dict)


class AppliedPricingStrategy(BaseModel):
    listing_id: str
    location_name: str
    pricing_strategy: Literal[
        "custom", "algorithm", "market-luxury", "market-upscale",
        "market-midscale", "market-economy"
    ]
    base_price: float = Field(..., ge=0)
    min_price: float = Field(..., ge=0)
    max_price: float = Field(..., ge=0)


class GetActivePricingProfileResponse(BaseModel):
    listing_id: str
    location_name: str
    active_pricing_profile: Optional[str] = None
    active_pricing_details: Optional[Dict[str, Any]] = None


# Models for specific date pricing
class SpecificDatePricingRequest(BaseModel):
    listing_id: str = Field(..., alias="listingId")
    location_name: str = Field(..., alias="locationName")
    start_date: str = Field(..., alias="startDate", description="Start date in YYYY-MM-DD format.")
    end_date: str = Field(..., alias="endDate", description="End date in YYYY-MM-DD format.")


class SpecificDatePricingResponse(BaseModel):
    listing_id: str
    location_name: str
    requested_date_range: Dict[str, str]
    market_data_source_table: Optional[str] = None
    processed_base_price: Optional[float] = None
    market_range_base_price: Optional[float] = None
    base_percentage_difference: Optional[float] = None
    daily_percentage_adjustments: Dict[str, float]  # Date -> Percentage Adjustment
    message: Optional[str] = None


class AllSpecificDatePricingResponse(BaseModel):
    listing_id: str
    location_name: str
    all_specific_date_adjustments: Dict[str, float]


# --- Global Variables ---
active_background_tasks: Dict[str, asyncio.Task] = {}


# --- Database Helper Functions ---
def sanitize_location_name_for_table(location_name: str) -> str:
    if not location_name or not isinstance(location_name, str): return "invalid_location"
    name = re.sub(r'\s+', '_', location_name.strip().lower())
    name = re.sub(r'[^a-zA-Z0-9_]', '', name)
    if re.match(r'^[0-9_]+$', name) or (name and name[0].isdigit()): name = f"loc_{name}"
    return name if name else "default_location"


def get_market_table_name(location_name_raw: str) -> str:
    sanitized_prefix = sanitize_location_name_for_table(location_name_raw)
    return f"{sanitized_prefix}_market"


def get_central_db_connection() -> mysql.connector.MySQLConnection:
    try:
        temp_conn_config = {**BASE_DB_CONFIG}
        temp_conn = mysql.connector.connect(**temp_conn_config)
        cursor = temp_conn.cursor()
        cursor.execute(
            f"CREATE DATABASE IF NOT EXISTS `{CENTRAL_DB_NAME}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
        cursor.close();
        temp_conn.close()
        config = {**BASE_DB_CONFIG, 'database': CENTRAL_DB_NAME}
        conn = mysql.connector.connect(**config)
        return conn
    except MySQLError as e:
        print(f"DB: CRITICAL Error connecting to MySQL Database '{CENTRAL_DB_NAME}': {e}");
        traceback.print_exc()
        raise HTTPException(status_code=503, detail=f"Database connection error for {CENTRAL_DB_NAME}.")
    except Exception as e:
        print(f"DB: CRITICAL Non-MySQL error during DB connection setup: {e}");
        traceback.print_exc()
        raise HTTPException(status_code=503, detail="Unexpected error connecting to database.")


def ensure_table_exists(db_conn: mysql.connector.MySQLConnection, table_name: str, create_sql_template: str):
    cursor = None
    try:
        cursor = db_conn.cursor()
        cursor.execute(f"SHOW TABLES LIKE '{table_name}'")
        if not cursor.fetchone():
            create_sql = create_sql_template.format(
                table_name=table_name) if '{table_name}' in create_sql_template else create_sql_template
            cursor.execute(create_sql)
            print(f"DB: Table '{table_name}' created successfully.")
    except MySQLError as e:
        print(f"DB: Error checking or creating table '{table_name}': {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Failed to ensure table '{table_name}' exists.")
    finally:
        if cursor: cursor.close()


def ensure_required_columns_exist(db_conn: mysql.connector.MySQLConnection):
    table_name = USER_LISTING_TABLE_NAME
    cursor = None
    try:
        cursor = db_conn.cursor()
        cursor.execute(f"SHOW TABLES LIKE '{table_name}'")
        if not cursor.fetchone():
            print(f"DB: ERROR - Table '{table_name}' does not exist. Cannot add columns.")
            raise MySQLError(f"Prerequisite failed: Table '{table_name}' not found during column check.")
        cursor.execute(f"SHOW COLUMNS FROM `{table_name}`")
        existing_columns = {str(row[0]).lower().strip() for row in cursor.fetchall()}
        alter_statements = []
        for col_name, col_def in REQUIRED_COLUMNS_SCHEMA.items():
            if col_name.lower().strip() not in existing_columns:
                alter_statements.append(f"ALTER TABLE `{table_name}` ADD COLUMN `{col_name}` {col_def}")
        if alter_statements:
            print(f"DB: Executing {len(alter_statements)} ADD COLUMN statement(s) for '{table_name}'...")
            for statement in alter_statements:
                try:
                    cursor.execute(statement)
                except MySQLError as alter_err:
                    if alter_err.errno == MySQLError.ER_DUP_FIELDNAME:
                        print(f"DB: Warning - Column for statement '{statement}' already existed. Skipping.")
                    else:
                        print(f"DB: Error executing ALTER statement: {statement}. Error: {alter_err}"); raise alter_err
            print(f"DB: Schema modification(s) applied for '{table_name}'.")
    except MySQLError as e:
        print(f"DB: Error during schema check/modification for '{table_name}': {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Failed schema update for table '{table_name}'.")
    finally:
        if cursor: cursor.close()


def initialize_listing_defaults_in_db(db_conn: mysql.connector.MySQLConnection, listing_id: str):
    table_name = USER_LISTING_TABLE_NAME
    cursor = None
    try:
        cursor = db_conn.cursor(dictionary=True)
        query = f"SELECT listId, occupancyProfile, occupancyProfileDetails, minStayProfile, active_pricing_profile FROM `{table_name}` WHERE listId = %s"
        cursor.execute(query, (listing_id,))
        listing_data = cursor.fetchone()
        if not listing_data: return

        update_params = {}
        default_occ_profile = "default"
        default_occ_details = json.dumps(ALL_PREDEFINED_PRICING_PROFILES[default_occ_profile])
        current_occ_profile = listing_data.get('occupancyProfile')
        current_occ_details_str = listing_data.get('occupancyProfileDetails')
        valid_occ_details = False
        if current_occ_details_str:
            try:
                loaded = json.loads(current_occ_details_str)
                if isinstance(loaded, dict) and loaded: valid_occ_details = True
            except (TypeError, json.JSONDecodeError):
                pass
        if not current_occ_profile or current_occ_profile not in ALL_PREDEFINED_PRICING_PROFILES or not valid_occ_details:
            update_params['occupancyProfile'] = default_occ_profile
            update_params['occupancyProfileDetails'] = default_occ_details
        if not listing_data.get('minStayProfile'): update_params['minStayProfile'] = 'default'
        if not listing_data.get('active_pricing_profile'): update_params['active_pricing_profile'] = 'algorithmic'

        if update_params:
            set_clauses = ", ".join([f"`{key}` = %s" for key in update_params.keys()])
            sql_update = f"UPDATE `{table_name}` SET {set_clauses} WHERE listId = %s"
            cursor.execute(sql_update, list(update_params.values()) + [listing_id])
        _calculate_and_update_listing_daily_adjustments_in_db(listing_id, db_conn, cursor_to_use=cursor)
    except MySQLError as e:
        print(f"DB: MySQL Error during default init for {listing_id}: {e}"); traceback.print_exc(); raise
    except Exception as e:
        print(f"DB: General Error during default init for {listing_id}: {e}"); traceback.print_exc(); raise
    finally:
        if cursor:
            try:
                cursor.close()
            except Exception as close_err:
                print(f"DB: Warning - Error closing cursor in init_listing_defaults: {close_err}")


# --- Helper Functions (Calculation Logic & General Purpose) ---
def print_system_usage():
    try:
        cpu = psutil.cpu_percent(interval=0.1);
        mem = psutil.virtual_memory()
        # print(f"SYS: CPU: {cpu:.1f}% | RAM: {mem.used/(1024**3):.2f}/{mem.total/(1024**3):.2f}GB ({mem.percent}%)")
    except Exception as e:
        print(f"SYS: Warning - Could not get system usage: {e}")


def timestamp_to_str(obj: Any) -> Any:
    if isinstance(obj, (datetime, date, pd.Timestamp)):
        try:
            return obj.strftime('%Y-%m-%d') if pd.notna(obj) else None
        except ValueError:
            return str(obj)
    if isinstance(obj, list): return [timestamp_to_str(item) for item in obj]
    if isinstance(obj, dict): return {k: timestamp_to_str(v) for k, v in obj.items()}
    if hasattr(obj, 'dtype') and pd.api.types.is_datetime64_any_dtype(obj.dtype):
        try:
            return pd.Timestamp(obj).strftime('%Y-%m-%d') if pd.notna(obj) else None
        except Exception:
            return str(obj)
    return obj


def parse_available_dates_column(date_input: Any) -> List[date]:
    if pd.isna(date_input): return []
    parsed_dates: List[pd.Timestamp] = []
    if isinstance(date_input, (list, tuple)):
        items = [d for d in date_input if isinstance(d, (str, datetime, pd.Timestamp, date)) and not pd.isna(d)]
        str_items = [item.strftime('%Y-%m-%d') if isinstance(item, (datetime, date, pd.Timestamp)) else str(item) for
                     item in items]
        parsed_dates = [pd.to_datetime(d, errors='coerce') for d in str_items]
        return sorted(list(set(ts.date() for ts in parsed_dates if pd.notna(ts))))

    input_str = str(date_input).strip()
    if not input_str or input_str.lower() in ['[]', '{}', 'nan', 'nat', 'none', 'null', "''", '""']: return []
    if input_str.endswith(';'): input_str = input_str[:-1].strip()
    try:
        if input_str.startswith('[') and input_str.endswith(']'):
            clean_str = re.sub(r'\s*,\s*', ',', input_str.replace('\n', '').replace('\r', ''))
            clean_str = re.sub(r'\[\s+', '[', clean_str);
            clean_str = re.sub(r'\s+\]', ']', clean_str)
            try:
                data = json.loads(clean_str)
            except json.JSONDecodeError:
                data = ast.literal_eval(clean_str)
            if isinstance(data, list):
                parsed_dates = [pd.to_datetime(d, errors='coerce') for d in data if isinstance(d, str) and d]
        elif input_str.startswith('DatetimeIndex(['):
            dates_found = re.findall(r"'(\d{4}-\d{2}-\d{2})'", input_str) or re.findall(r"(\d{4}-\d{2}-\d{2})",
                                                                                        input_str)
            parsed_dates = [pd.to_datetime(d, errors='coerce') for d in dates_found]
        elif ',' in input_str:
            parsed_dates = [pd.to_datetime(d.strip(), errors='coerce') for d in input_str.split(',') if d.strip()]
        else:
            single_dt = pd.to_datetime(input_str, errors='coerce')
            if pd.notna(single_dt): parsed_dates = [single_dt]
        return sorted(list(set(ts.date() for ts in parsed_dates if pd.notna(ts))))
    except Exception as e:
        print(
            f"CRITICAL Error in parse_available_dates_column (type: {type(date_input)}, value: '{str(date_input)[:100]}...'): {e}"); traceback.print_exc(); return []


def _calculate_occupancy_for_window(avail_set: Set[date], start_dt: date, end_dt: date) -> float:
    if start_dt > end_dt: return 0.0
    total_days = (end_dt - start_dt).days + 1
    if total_days <= 0: return 0.0
    avail_count = sum(1 for i in range(total_days) if (start_dt + timedelta(days=i)) in avail_set) if avail_set else 0
    occupied = total_days - avail_count
    return round(max(0.0, min(100.0, (occupied / total_days) * 100 if total_days > 0 else 0.0)), 2)


def get_occupancy_tier_key(occupancy_percentage: float) -> str:
    occ = max(0.0, min(100.0, occupancy_percentage))
    if occ <= 10: return "≤10"
    if occ <= 20: return "≤20"
    if occ <= 30: return "≤30"
    if occ <= 40: return "≤40"
    if occ <= 50: return "≤50"
    if occ <= 60: return "≤60"
    if occ <= 70: return "≤70"
    if occ <= 80: return "≤80"
    return "≤100"


def _calculate_daily_occupancy_adjustments_for_listing_logic(
        avail_dates: List[date], enabled: bool, profile_details: Dict[str, Dict[str, Any]]
) -> Dict[str, int]:
    adjustments: Dict[str, int] = {}
    today = date.today();
    limit_days = 90
    if not enabled or not isinstance(profile_details, dict) or not profile_details:
        for i in range(limit_days): adjustments[(today + timedelta(days=i)).strftime("%Y-%m-%d")] = 0
        return adjustments
    avail_set = set(avail_dates)
    for i in range(limit_days):
        target_dt = today + timedelta(days=i);
        target_dt_str = target_dt.strftime("%Y-%m-%d");
        adj = 0
        win_key = None
        if 0 <= i <= 15:
            win_key = "0_15_days"
        elif 16 <= i <= 30:
            win_key = "16_30_days"
        elif 31 <= i <= 60:
            win_key = "30_60_days"
        elif 61 <= i < limit_days:
            win_key = "60_90_days"
        if win_key and win_key in profile_details:
            win_def = OCCUPANCY_PROFILE_WINDOWS.get(win_key)
            prof_for_win = profile_details.get(win_key)
            if win_def and isinstance(prof_for_win, dict):
                win_start = today + timedelta(days=win_def[0]);
                win_end = today + timedelta(days=win_def[1])
                win_occ = _calculate_occupancy_for_window(avail_set, win_start, win_end)
                occ_tier = get_occupancy_tier_key(win_occ)
                adj = prof_for_win.get(occ_tier, 0)
                try:
                    adj = int(adj)
                except (ValueError, TypeError):
                    adj = 0
        adjustments[target_dt_str] = adj
    return adjustments


def _calculate_and_update_listing_daily_adjustments_in_db(
        listing_id: str, db_conn: mysql.connector.MySQLConnection,
        cursor_to_use: Optional[mysql.connector.cursor.MySQLCursorDict] = None
):
    table = USER_LISTING_TABLE_NAME
    cur, close_cur = (db_conn.cursor(dictionary=True), True) if cursor_to_use is None else (cursor_to_use, False)
    try:
        if not cur: return
        cur.execute(
            f"SELECT available_dates, occupancyEnabled, occupancyProfile, occupancyProfileDetails FROM `{table}` WHERE listId = %s",
            (listing_id,))
        data = cur.fetchone()
        if not data: return
        avail_dates = parse_available_dates_column(data.get('available_dates'))
        enabled = data.get('occupancyEnabled', True)
        profile_name = data.get('occupancyProfile', 'default')
        profile_details_str = data.get('occupancyProfileDetails')
        active_details = {}
        if profile_details_str:
            try:
                loaded = json.loads(profile_details_str)
                if isinstance(loaded, dict) and loaded:
                    active_details = loaded
                else:
                    active_details = ALL_PREDEFINED_PRICING_PROFILES.get(profile_name,
                                                                         ALL_PREDEFINED_PRICING_PROFILES['default'])
            except (TypeError, json.JSONDecodeError):
                active_details = ALL_PREDEFINED_PRICING_PROFILES.get(profile_name,
                                                                     ALL_PREDEFINED_PRICING_PROFILES['default'])
        else:
            active_details = ALL_PREDEFINED_PRICING_PROFILES.get(profile_name,
                                                                 ALL_PREDEFINED_PRICING_PROFILES['default'])
        adjustments = _calculate_daily_occupancy_adjustments_for_listing_logic(avail_dates, enabled, active_details)
        cur.execute(f"UPDATE `{table}` SET daily_occupancy_adjustments_json = %s WHERE listId = %s",
                    (json.dumps(adjustments), listing_id))
    except MySQLError as e:
        print(f"DB: MySQLError daily adj for {listing_id}: {e}"); traceback.print_exc(); raise
    except Exception as e:
        print(f"DB: General Error daily adj for {listing_id}: {e}"); traceback.print_exc(); raise
    finally:
        if close_cur and cur: cur.close()


# def _calculate_and_update_calendar_pricing_in_db(
#         listing_id: str, db_conn: mysql.connector.MySQLConnection,
#         cursor_to_use: Optional[mysql.connector.cursor.MySQLCursorDict] = None
# ):
#     table = USER_LISTING_TABLE_NAME
#     cur, close_cur = (db_conn.cursor(dictionary=True), True) if cursor_to_use is None else (cursor_to_use, False)
#     if not cur: print(
#         f"DB CALENDAR: FATAL - Failed to obtain cursor for calendar pricing update ({listing_id})."); return
#     calendar_prices = {};
#     error_msg = None
#     try:
#         cur.execute(f"SELECT active_pricing_details, processed_pricing_json FROM `{table}` WHERE listId = %s",
#                     (listing_id,))
#         data = cur.fetchone()
#         if not data: print(f"DB CALENDAR: Listing '{listing_id}' not found for calendar pricing. Skipping."); return
#         base_p, min_p, max_p = None, None, None
#         active_details_str = data.get('active_pricing_details')
#         if active_details_str:
#             try:
#                 details = json.loads(active_details_str)
#                 if isinstance(details, dict):
#                     base_p = float(details['base_price']) if details.get('base_price') is not None else None
#                     min_p = float(details['min_price']) if details.get('min_price') is not None else None
#                     max_p = float(details['max_price']) if details.get('max_price') is not None else None
#             except (json.JSONDecodeError, ValueError, TypeError, KeyError):
#                 pass
#         if base_p is None:
#             processed_pricing_str = data.get('processed_pricing_json')
#             if processed_pricing_str:
#                 try:
#                     details = json.loads(processed_pricing_str)
#                     if isinstance(details, dict) and not details.get("error"):
#                         base_p = float(details['base_price']) if details.get('base_price') is not None else None
#                         min_p = float(details['min_price']) if details.get('min_price') is not None else None
#                         max_p = float(details['max_price']) if details.get('max_price') is not None else None
#                 except (json.JSONDecodeError, ValueError, TypeError, KeyError):
#                     pass
#         if base_p is None or base_p <= 0:
#             error_msg = "Valid base price not determined."
#         else:
#             if min_p is None or min_p <= 0: min_p = base_p * 0.4
#             if max_p is None or max_p <= min_p: max_p = base_p * 2.0
#             min_p = max(0.01, min_p);
#             max_p = max(min_p, max_p)
#             today = date.today();
#             start_calc = date(today.year, today.month, 1)
#             current_month_loop = start_calc
#             for _ in range(12):
#                 days_in_month = monthrange(current_month_loop.year, current_month_loop.month)[1]
#                 for day_offset in range(days_in_month):
#                     current_day = date(current_month_loop.year, current_month_loop.month, day_offset + 1)
#                     day_price = base_p
#                     if current_day.weekday() in [SATURDAY, SUNDAY]:
#                         day_price = max_p * random.uniform(0.40, 0.7) if max_p > base_p else base_p * random.uniform(
#                             1.10, 1.25)
#                     else:
#                         day_price = base_p * (1 + random.choice([-1, 1]) * random.uniform(0.05, 0.10))
#                     day_price = round(max(0.01, min(max_p, max(min_p, day_price))), 2)
#                     calendar_prices[current_day.strftime("%Y-%m-%d")] = day_price
#                 if current_month_loop.month == 12:
#                     current_month_loop = date(current_month_loop.year + 1, 1, 1)
#                 else:
#                     current_month_loop = date(current_month_loop.year, current_month_loop.month + 1, 1)
#         output_json = json.dumps({"error": error_msg, "data": {}}) if error_msg else json.dumps(calendar_prices)
#         cur.execute(f"UPDATE `{table}` SET calendar_pricing_json = %s WHERE listId = %s", (output_json, listing_id))
#     except MySQLError as e:
#         print(f"DB CALENDAR: MySQLError for {listing_id}: {e}"); traceback.print_exc(); raise
#     except Exception as e:
#         print(f"DB CALENDAR: General Error for {listing_id}: {e}"); traceback.print_exc(); raise
#     finally:
#         if close_cur and cur: cur.close()


# def _calculate_and_update_calendar_pricing_in_db(
#         listing_id: str, db_conn: mysql.connector.MySQLConnection,
#         cursor_to_use: Optional[mysql.connector.cursor.MySQLCursorDict] = None
# ):
#     table = USER_LISTING_TABLE_NAME
#     cur, close_cur = (db_conn.cursor(dictionary=True), True) if cursor_to_use is None else (cursor_to_use, False)
#     if not cur:
#         print(f"DB CALENDAR: FATAL - Failed to obtain cursor for calendar pricing update ({listing_id}).")
#         return
#
#     calendar_prices = {}
#     error_msg = None
#     base_p, min_p_source, max_p_source = None, None, None  # min_p_source/max_p_source are from the same JSON as base_p
#
#     try:
#         cur.execute(
#             f"SELECT active_pricing_details, processed_pricing_json FROM `{table}` WHERE listId = %s",
#             (listing_id,)
#         )
#         data = cur.fetchone()
#         if not data:
#             print(f"DB CALENDAR: Listing '{listing_id}' not found for calendar pricing. Skipping.")
#             return
#
#         # 1. Try to get base price from active_pricing_details
#         active_details_str = data.get('active_pricing_details')
#         if active_details_str:
#             try:
#                 details = json.loads(active_details_str)
#                 if isinstance(details, dict):
#                     potential_base_val = details.get('base_price')
#                     if potential_base_val is not None:
#                         parsed_val = float(potential_base_val)
#                         if parsed_val > 0:  # Must be a positive value
#                             base_p = parsed_val
#                             # If base_p is from here, try to get min/max from here too
#                             if details.get('min_price') is not None:
#                                 min_p_source = float(details['min_price'])
#                             if details.get('max_price') is not None:
#                                 max_p_source = float(details['max_price'])
#             except (json.JSONDecodeError, ValueError, TypeError, KeyError):
#                 # If parsing fails or keys missing, base_p remains None
#                 pass
#
#         # 2. If base_p not found or not positive from active_details, try processed_pricing_json
#         #    This is more lenient: will use base_price if it's valid, even if an "error" field exists.
#         if base_p is None or base_p <= 0:
#             processed_pricing_str = data.get('processed_pricing_json')
#             if processed_pricing_str:
#                 try:
#                     details = json.loads(processed_pricing_str)
#                     if isinstance(details, dict):
#                         potential_base_val = details.get('base_price')
#                         if potential_base_val is not None:
#                             parsed_val = float(potential_base_val)
#                             if parsed_val > 0:  # Must be a positive value
#                                 base_p = parsed_val
#                                 min_p_source, max_p_source = None, None  # Reset, take from this source
#                                 # If base_p is from here, try to get min/max from here too
#                                 if details.get('min_price') is not None:
#                                     min_p_source = float(details['min_price'])
#                                 if details.get('max_price') is not None:
#                                     max_p_source = float(details['max_price'])
#                                 # If parsed_val was not > 0, base_p remains as it was (e.g., None)
#                 except (json.JSONDecodeError, ValueError, TypeError, KeyError):
#                     # If parsing fails, base_p remains as it was
#                     pass
#
#         # 3. Final check and default min/max if necessary
#         if base_p is None or base_p <= 0:
#             error_msg = "Valid base price not determined."
#         else:
#             # Use min_p_source and max_p_source if they were successfully parsed
#             # from the same source as base_p, otherwise default them.
#             final_min_p = min_p_source
#             final_max_p = max_p_source
#
#             if final_min_p is None or final_min_p <= 0:
#                 final_min_p = base_p * 0.4
#             if final_max_p is None or final_max_p <= final_min_p:  # max must be > min
#                 final_max_p = base_p * 2.0
#
#             final_min_p = round(max(0.01, final_min_p), 2)  # Ensure min price is at least 0.01
#             final_max_p = round(max(final_min_p, final_max_p), 2)  # Ensure max price is at least min_price
#
#             today = date.today()
#             start_calc = date(today.year, today.month, 1)  # Start from the beginning of the current month
#             current_month_loop = start_calc
#
#             for _ in range(12):  # Generate for 12 months
#                 days_in_month = monthrange(current_month_loop.year, current_month_loop.month)[1]
#                 for day_offset in range(days_in_month):
#                     current_day = date(current_month_loop.year, current_month_loop.month, day_offset + 1)
#
#                     # Basic seasonality/weekend logic (can be expanded)
#                     day_price = base_p
#                     if current_day.weekday() in [SATURDAY, SUNDAY]:  # SATURDAY=5, SUNDAY=6
#                         # Example: Weekends are 10-25% higher than base, but capped by max_price
#                         day_price = base_p * random.uniform(1.10, 1.25)
#                     else:
#                         # Example: Weekdays vary +/- 5-10% around base
#                         day_price = base_p * (1 + random.choice([-1, 1]) * random.uniform(0.05, 0.10))
#
#                     # Apply min/max bounds and round
#                     day_price = round(max(0.01, min(final_max_p, max(final_min_p, day_price))), 2)
#                     calendar_prices[current_day.strftime("%Y-%m-%d")] = day_price
#
#                 # Move to the next month
#                 if current_month_loop.month == 12:
#                     current_month_loop = date(current_month_loop.year + 1, 1, 1)
#                 else:
#                     current_month_loop = date(current_month_loop.year, current_month_loop.month + 1, 1)
#
#         output_json = json.dumps({"error": error_msg, "data": {}}) if error_msg else json.dumps(calendar_prices)
#         cur.execute(f"UPDATE `{table}` SET calendar_pricing_json = %s WHERE listId = %s", (output_json, listing_id))
#
#     except MySQLError as e:
#         print(f"DB CALENDAR: MySQLError for {listing_id}: {e}");
#         traceback.print_exc()
#         # Avoid raising here if cursor_to_use is passed, let the caller handle commit/rollback
#         if close_cur:
#             raise
#         else:  # Log and potentially store error if we are a sub-operation
#             try:
#                 err_out = json.dumps({"error": f"DB error during calendar pricing: {e}", "data": {}})
#                 cur.execute(f"UPDATE `{table}` SET calendar_pricing_json = %s WHERE listId = %s", (err_out, listing_id))
#             except Exception as inner_e:
#                 print(
#                     f"DB CALENDAR: CRITICAL - Failed to write error to calendar_pricing_json for {listing_id}: {inner_e}")
#
#     except Exception as e:
#         print(f"DB CALENDAR: General Error for {listing_id}: {e}");
#         traceback.print_exc()
#         if close_cur:
#             raise
#         else:
#             try:
#                 err_out = json.dumps({"error": f"General error during calendar pricing: {e}", "data": {}})
#                 cur.execute(f"UPDATE `{table}` SET calendar_pricing_json = %s WHERE listId = %s", (err_out, listing_id))
#             except Exception as inner_e:
#                 print(
#                     f"DB CALENDAR: CRITICAL - Failed to write error to calendar_pricing_json for {listing_id}: {inner_e}")
#     finally:
#         if close_cur and cur:
#             try:
#                 cur.close()
#             except Exception as close_err:
#                 print(f"DB CALENDAR: Warning - Error closing cursor: {close_err}")


def _calculate_and_update_calendar_pricing_in_db(
        listing_id: str, db_conn: mysql.connector.MySQLConnection,
        cursor_to_use: Optional[mysql.connector.cursor.MySQLCursorDict] = None
):
    table = USER_LISTING_TABLE_NAME
    cur, close_cur = (db_conn.cursor(dictionary=True), True) if cursor_to_use is None else (cursor_to_use, False)
    if not cur:
        print(f"DB CALENDAR ({listing_id}): FATAL - Failed to obtain cursor.")
        return

    calendar_prices = {}
    error_msg = None
    base_p, min_p_source, max_p_source = None, None, None

    try:
        # Fetch all necessary data in one go
        cur.execute(
            f"SELECT active_pricing_details, processed_pricing_json, daily_occupancy_adjustments_json, specific_date_pricing_json FROM `{table}` WHERE listId = %s",
            (listing_id,)
        )
        data = cur.fetchone()
        if not data:
            print(f"DB CALENDAR ({listing_id}): Listing not found. Skipping.")
            return

        # 1. Determine Base Price (base_p), Min Price, Max Price
        active_details_str = data.get('active_pricing_details')
        if active_details_str:
            try:
                details = json.loads(active_details_str)
                if isinstance(details, dict):
                    potential_base_val = details.get('base_price')
                    if potential_base_val is not None:
                        parsed_val = float(potential_base_val)
                        if parsed_val > 0:
                            base_p = parsed_val
                            if details.get('min_price') is not None: min_p_source = float(details['min_price'])
                            if details.get('max_price') is not None: max_p_source = float(details['max_price'])
            except (json.JSONDecodeError, ValueError, TypeError, KeyError):
                pass

        if base_p is None or base_p <= 0:
            processed_pricing_str = data.get('processed_pricing_json')
            if processed_pricing_str:
                try:
                    details = json.loads(processed_pricing_str)
                    if isinstance(details, dict) and not details.get("error"):  # Check for error key
                        potential_base_val = details.get('base_price')
                        if potential_base_val is not None:
                            parsed_val = float(potential_base_val)
                            if parsed_val > 0:
                                base_p = parsed_val
                                min_p_source, max_p_source = None, None  # Reset, take from this source
                                if details.get('min_price') is not None: min_p_source = float(details['min_price'])
                                if details.get('max_price') is not None: max_p_source = float(details['max_price'])
                except (json.JSONDecodeError, ValueError, TypeError, KeyError):
                    pass

        # Fallback if still no base_p (e.g. first run, algo failed, no active set)
        if base_p is None or base_p <= 0:
            # Use the DEFAULT_FALLBACK_BASE_PRICE from process_data_and_save_all_to_db
            # or define it here if not accessible. For now, let's assume a default.
            DEFAULT_FALLBACK_BASE_PRICE_CALENDAR = 100.0
            print(
                f"DB CALENDAR ({listing_id}): Valid base price not found. Using fallback {DEFAULT_FALLBACK_BASE_PRICE_CALENDAR}.")
            base_p = DEFAULT_FALLBACK_BASE_PRICE_CALENDAR
            # min_p_source and max_p_source will be defaulted based on this fallback base_p

        # Finalize min/max prices
        final_min_p = min_p_source
        final_max_p = max_p_source
        if final_min_p is None or final_min_p <= 0: final_min_p = base_p * 0.4
        if final_max_p is None or final_max_p <= final_min_p: final_max_p = base_p * 2.0
        final_min_p = round(max(0.01, final_min_p), 2)
        final_max_p = round(max(final_min_p, final_max_p), 2)

        # 2. Load Adjustment Data
        daily_occ_adjustments: Dict[str, float] = {}  # Expects date_str -> percentage_float
        specific_date_adj_full: Dict[str, Dict[str, float]] = {}  # Expects date_str -> {"adj_perc": %, "mkt_price": $}

        if data.get('daily_occupancy_adjustments_json'):
            try:
                daily_occ_adjustments = json.loads(data['daily_occupancy_adjustments_json'])
            except json.JSONDecodeError:
                pass

        if data.get('specific_date_pricing_json'):
            try:
                loaded_specific = json.loads(data['specific_date_pricing_json'])
                if isinstance(loaded_specific, dict):  # Ensure it's a dict
                    specific_date_adj_full = loaded_specific
            except json.JSONDecodeError:
                pass

        # 3. Generate Calendar Prices
        today = date.today()
        start_calc = date(today.year, today.month, 1)
        current_month_loop = start_calc

        for _ in range(12):  # Generate for 12 months
            days_in_month = monthrange(current_month_loop.year, current_month_loop.month)[1]
            for day_offset in range(days_in_month):
                current_day = date(current_month_loop.year, current_month_loop.month, day_offset + 1)
                current_day_str = current_day.strftime("%Y-%m-%d")

                adjusted_price = base_p  # Start with the determined base price

                # Apply daily occupancy adjustment (percentage)
                occupancy_adj_percent = daily_occ_adjustments.get(current_day_str, 0.0)
                try:
                    adjusted_price = adjusted_price * (1 + float(occupancy_adj_percent) / 100.0)
                except (ValueError, TypeError):
                    print(f"DB CALENDAR ({listing_id}): Invalid occupancy adj for {current_day_str}. Skipping.")

                # Apply specific date market adjustment (percentage)
                # This percentage is applied to the price *after* occupancy adjustment.
                # Or, if specific_date_adj should be dominant, apply it to base_p directly.
                # Current logic: compounds on occupancy-adjusted price.
                specific_market_adj_entry = specific_date_adj_full.get(current_day_str)
                specific_market_adj_percent_val = None

                if isinstance(specific_market_adj_entry, dict) and "adj_perc" in specific_market_adj_entry:
                    try:
                        specific_market_adj_percent_val = float(specific_market_adj_entry["adj_perc"])
                    except (ValueError, TypeError):
                        print(
                            f"DB CALENDAR ({listing_id}): Invalid specific_date adj_perc for {current_day_str}. Skipping specific adj.")
                elif isinstance(specific_market_adj_entry, (int, float)):  # Handle old format if encountered
                    specific_market_adj_percent_val = float(specific_market_adj_entry)
                    print(f"DB CALENDAR ({listing_id}): Old format specific_date_adj for {current_day_str} used.")

                if specific_market_adj_percent_val is not None:
                    adjusted_price = adjusted_price * (1 + specific_market_adj_percent_val / 100.0)
                else:
                    # If no specific market adjustment, apply general rules (weekend, etc.)
                    if current_day.weekday() in [SATURDAY, SUNDAY]:
                        adjusted_price = adjusted_price * random.uniform(1.10, 1.25)
                    else:
                        adjusted_price = adjusted_price * (1 + random.choice([-1, 1]) * random.uniform(0.05, 0.10))

                # Apply min/max bounds
                final_day_price = round(max(0.01, min(final_max_p, max(final_min_p, adjusted_price))), 2)
                calendar_prices[current_day_str] = final_day_price

            if current_month_loop.month == 12:
                current_month_loop = date(current_month_loop.year + 1, 1, 1)
            else:
                current_month_loop = date(current_month_loop.year, current_month_loop.month + 1, 1)

        output_json = json.dumps(calendar_prices)  # No error message if base_p was defaulted

    except MySQLError as e:
        print(f"DB CALENDAR ({listing_id}): MySQLError: {e}");
        traceback.print_exc()
        error_msg = f"DB error: {e}"
        output_json = json.dumps({"error": error_msg, "data": {}})
        if close_cur: raise  # Re-raise if we own the cursor management
    except Exception as e:
        print(f"DB CALENDAR ({listing_id}): General Error: {e}");
        traceback.print_exc()
        error_msg = f"General error: {e}"
        output_json = json.dumps({"error": error_msg, "data": {}})
        if close_cur: raise  # Re-raise

    finally:
        # Always try to update the calendar_pricing_json, even if it's an error dict
        if 'output_json' in locals():  # Ensure output_json is defined
            try:
                cur.execute(f"UPDATE `{table}` SET calendar_pricing_json = %s WHERE listId = %s",
                            (output_json, listing_id))
                # Commit is handled by the caller if cursor_to_use is passed, or here if close_cur is True
                if close_cur and db_conn.is_connected():  # Only commit if we opened the connection/cursor here
                    db_conn.commit()
            except Exception as update_err:
                print(f"DB CALENDAR ({listing_id}): CRITICAL - Failed to write to calendar_pricing_json: {update_err}")

        if close_cur and cur:
            try:
                cur.close()
            except Exception as close_err:
                print(f"DB CALENDAR ({listing_id}): Warning - Error closing cursor: {close_err}")
def generate_market_pricing(df_market: pd.DataFrame) -> Dict[str, Any]:
    res: Dict[str, Any] = {"luxury": {}, "upscale": {}, "midscale": {}, "economy": {}, "error": None}
    if df_market is None or df_market.empty: res["error"] = "No market data"; return res
    if 'price' not in df_market.columns: res["error"] = "Missing 'price' column"; return res
    prices = pd.to_numeric(df_market['price'], errors='coerce').dropna()
    if prices.empty or len(prices) < 4: res["error"] = f"Insufficient prices ({len(prices)})"; return res
    try:
        p90, p75, p50, p35 = prices.quantile([0.90, 0.75, 0.50, 0.35]).tolist()
        mult_max, mult_min = 2.0, 0.4
        for seg, p_val in [("luxury", p90), ("upscale", p75), ("midscale", p50), ("economy", p35)]:
            res[seg] = {"base_price": round(float(p_val), 2), "max_price": round(float(p_val * mult_max), 2),
                        "min_price": round(float(p_val * mult_min), 2)}
    except Exception as e:
        res["error"] = f"Error: {e}"; traceback.print_exc()
    return res


def _calculate_monthly_occupancy_for_listing(avail_dates: List[date], month_keys: List[str]) -> Dict[
    str, Optional[float]]:
    occ_data: Dict[str, Optional[float]] = {key: None for key in month_keys}
    if not avail_dates: return occ_data
    avail_set = set(avail_dates)
    if not avail_set: return occ_data
    try:
        min(avail_set); max(avail_set)
    except ValueError:
        return occ_data
    for key in month_keys:
        try:
            dt = datetime.strptime(key, "%b_%Y");
            year, month = dt.year, dt.month
            start_dt = date(year, month, 1);
            days_in_m = monthrange(year, month)[1]
            avail_count = sum(1 for day_num in range(1, days_in_m + 1) if date(year, month, day_num) in avail_set)
            occupied = days_in_m - avail_count
            occ_data[key] = round(max(0.0, min(100.0, (occupied / days_in_m) * 100 if days_in_m > 0 else 0.0)), 2)
        except Exception:
            pass
    return occ_data


def calculate_monthly_averages(prop_df: Optional[pd.DataFrame], market_df: Optional[pd.DataFrame]) -> Dict[
    str, Dict[str, Optional[float]]]:
    prop_occ: Dict[str, Optional[float]] = {};
    market_avg: Dict[str, Optional[float]] = {}
    keys: List[str] = []
    if market_df is not None and not market_df.empty:
        pot_keys = [col for col in market_df.columns if re.match(r'^[A-Z][a-z]{2}_\d{4}$', col)]
        if pot_keys:
            try:
                keys = sorted(pot_keys, key=lambda x: datetime.strptime(x, "%b_%Y"))
            except ValueError:
                keys = pot_keys
    if not keys:
        start_m = date(datetime.today().year, datetime.today().month, 1)
        keys = [(start_m + pd.DateOffset(months=i)).strftime("%b_%Y") for i in range(24)]
    if prop_df is not None and not prop_df.empty and 'available_dates' in prop_df.columns:
        parsed_dates = parse_available_dates_column(prop_df['available_dates'].iloc[0])
        prop_occ = _calculate_monthly_occupancy_for_listing(parsed_dates, keys) if parsed_dates else {k: None for k in
                                                                                                      keys}
    else:
        prop_occ = {k: None for k in keys}
    if market_df is not None and not market_df.empty:
        for col in keys:
            if col in market_df.columns:
                try:
                    avg = pd.to_numeric(market_df[col], errors='coerce').mean()
                    market_avg[col] = round(float(avg), 2) if pd.notna(avg) else None
                except Exception:
                    market_avg[col] = None
            else:
                market_avg[col] = None
    else:
        market_avg = {k: None for k in keys}
    return {"property": prop_occ, "market": market_avg}


def apply_percentage_variation(base_percentage: float, variation_range_points: tuple = (5.0, 10.0)) -> float:
    # Applies an additive percentage point variation
    variation_magnitude = random.uniform(variation_range_points[0], variation_range_points[1])
    variation_direction = random.choice([-1, 1])
    return round(base_percentage + (variation_magnitude * variation_direction), 2)


# --- Lifespan Function ---
@asynccontextmanager
async def lifespan(app: FastAPI):
    print("APP: FastAPI application startup sequence initiated.")
    conn = None
    try:
        conn = get_central_db_connection()
        cursor = conn.cursor();
        cursor.execute("SELECT 1");
        cursor.fetchone();
        cursor.close()
        print(f"APP: Successfully verified connection to central DB '{CENTRAL_DB_NAME}'.")
    except HTTPException as http_exc:
        print(f"APP: CRITICAL STARTUP FAILURE - DB: {http_exc.detail}")
    except Exception as e:
        print(f"APP: CRITICAL STARTUP FAILURE - Unexpected DB error: {e}"); traceback.print_exc()
    finally:
        if conn and conn.is_connected(): conn.close()
    print("APP: Startup sequence complete. Application ready.")
    yield
    print("APP: FastAPI application shutdown sequence initiated.")
    if active_background_tasks: print(f"APP: {len(active_background_tasks)} background task(s) might still be running.")
    print("APP: Shutdown sequence complete.")


# --- FastAPI App Setup ---
app = FastAPI(title="Dynamic Pricing API", version="1.4.0", lifespan=lifespan)  # Version bump for specific_date_pricing
app.add_middleware(CORSMiddleware, allow_origins=["*"], allow_credentials=True, allow_methods=["*"],
                   allow_headers=["*"])


# --- Main Data Processing Task (Background) ---
# async def process_data_and_save_all_to_db(listing_id: str, location_name_raw: str):
#     market_table = get_market_table_name(location_name_raw)
#     sanitized_loc = sanitize_location_name_for_table(location_name_raw)
#     task_id = f"{CENTRAL_DB_NAME}:{sanitized_loc}:{listing_id}"
#     print(f"TASK {task_id}: Background processing STARTED.")
#     print_system_usage()
#     conn, cursor, status = None, None, 'failed'
#     try:
#         conn = get_central_db_connection()
#         cursor = conn.cursor(dictionary=True)
#         cursor.execute(
#             f"UPDATE `{USER_LISTING_TABLE_NAME}` SET processing_status = %s, last_processed_at = %s WHERE listId = %s",
#             ('pending', datetime.now(), listing_id))
#         if cursor.rowcount == 0: print(
#             f"TASK {task_id}: Warning - Listing {listing_id} not found when setting to pending.")
#         conn.commit();
#         status = 'pending'
#         cursor.execute(f"SELECT *, active_pricing_profile FROM `{USER_LISTING_TABLE_NAME}` WHERE listId = %s",
#                        (listing_id,))
#         listing_dict = cursor.fetchone()
#         if not listing_dict: print(f"TASK {task_id}: Listing {listing_id} vanished. Aborting."); return
#         listing_df = pd.DataFrame([listing_dict])
#         active_profile = listing_dict.get('active_pricing_profile') or "algorithmic"
#         listing_lat, listing_long = listing_df.get('lat', pd.Series([None])).iloc[0], \
#         listing_df.get('long', pd.Series([None])).iloc[0]
#
#         try:
#             central_cfg = {**BASE_DB_CONFIG, 'database': CENTRAL_DB_NAME}
#             loop = asyncio.get_running_loop()
#             await loop.run_in_executor(None, functools.partial(sql_update_all_amenities.process_amenities_in_db,
#                                                                db_config_override=central_cfg,
#                                                                table_name_override=USER_LISTING_TABLE_NAME))
#         except Exception as amenities_err:
#             print(f"TASK {task_id}: Amenities update error: {amenities_err}")
#
#         market_df = pd.DataFrame()
#         cursor.execute(f"SHOW TABLES LIKE '{market_table}'")
#         if cursor.fetchone():
#             market_df = pd.read_sql_query(f"SELECT * FROM `{market_table}`", conn)
#         else:
#             print(f"TASK {task_id}: Market table '{market_table}' not found.")
#
#         market_200_df = pd.DataFrame()
#         if not market_df.empty and pd.notna(listing_lat) and pd.notna(listing_long):
#             try:
#                 market_200_df = process_dataframe(market_df, lat=listing_lat, long=listing_long)
#             except Exception as filter_err:
#                 print(f"TASK {task_id}: Market filter error: {filter_err}")
#
#         results = {}
#         try:
#             results["processed_amenities_json"] = json.dumps(
#                 timestamp_to_str(get_top_amenities(market_200_df, listing_df)))
#         except Exception as e:
#             results["processed_amenities_json"] = json.dumps({"error": f"Amenities: {e}"})
#
#         min_stay_profile = listing_dict.get('minStayProfile') or "default"
#         try:
#             min_stay_res = generate_recommendations(market_200_df, listing_df)
#             results["processed_min_stay_json"] = json.dumps(timestamp_to_str(min_stay_res))
#             if min_stay_profile == "default":
#                 parsed_rec = json.loads(results["processed_min_stay_json"])
#                 if not (isinstance(parsed_rec, dict) and parsed_rec.get("error")): results["customMinStayData"] = \
#                 results["processed_min_stay_json"]
#             if not listing_dict.get('minStayProfile'): results["minStayProfile"] = "default"
#         except Exception as e:
#             err_json = json.dumps({"error": f"Min Stay: {e}"})
#             results["processed_min_stay_json"] = err_json
#             if min_stay_profile == "default": results["customMinStayData"] = err_json
#             if not listing_dict.get('minStayProfile'): results["minStayProfile"] = "default"
#
#         try:
#             base_price_res = get_recommended_price(market_200_df, listing_df)
#             if base_price_res is not None and pd.notna(base_price_res):
#                 bp = float(base_price_res)
#                 pricing = {"base_price": round(bp, 2), "max_price": round(bp * 2.0, 2), "min_price": round(bp * 0.4, 2)}
#                 results["processed_pricing_json"] = json.dumps(pricing)
#                 if active_profile == "algorithmic":
#                     results["active_pricing_details"] = results["processed_pricing_json"]
#                 if not listing_dict.get('active_pricing_profile'): results["active_pricing_profile"] = "algorithmic"
#             else:
#                 raise ValueError("Base price N/A")
#         except Exception as e:
#             err_msg = f"Base Price: {e}"
#             err_json_details = json.dumps({"error": err_msg, "base_price": None, "min_price": None, "max_price": None})
#             results["processed_pricing_json"] = err_json_details
#             if active_profile == "algorithmic": results["active_pricing_details"] = err_json_details
#             if not listing_dict.get('active_pricing_profile'): results["active_pricing_profile"] = "algorithmic"
#
#         try:
#             results["processed_market_pricing_json"] = json.dumps(
#                 timestamp_to_str(generate_market_pricing(market_200_df)))
#         except Exception as e:
#             results["processed_market_pricing_json"] = json.dumps({"error": f"Market Tiers: {e}"})
#         try:
#             results["processed_monthly_averages_json"] = json.dumps(
#                 timestamp_to_str(calculate_monthly_averages(listing_df, market_200_df)))
#         except Exception as e:
#             results["processed_monthly_averages_json"] = json.dumps({"error": f"Monthly Avg: {e}"})
#         try:
#             map_cols = ['price', 'lat', 'long', 'color']
#             if not market_200_df.empty and all(c in market_200_df for c in map_cols):
#                 results["processed_map_recs_json"] = json.dumps(
#                     timestamp_to_str(market_200_df[map_cols].dropna().to_dict('records')))
#             else:
#                 results["processed_map_recs_json"] = json.dumps([])
#         except Exception as e:
#             results["processed_map_recs_json"] = json.dumps({"error": f"Map Recs: {e}"})
#
#         try:
#             _calculate_and_update_listing_daily_adjustments_in_db(listing_id, conn, cursor_to_use=cursor)
#         except Exception as e:
#             print(f"TASK {task_id}: Daily adjustment step error: {e}")
#         try:
#             _calculate_and_update_calendar_pricing_in_db(listing_id, conn, cursor_to_use=cursor)
#         except Exception as e:
#             print(f"TASK {task_id}: ERROR during calendar pricing step: {e}")
#
#         set_clauses, params = [], []
#         for key, value in results.items(): set_clauses.append(f"`{key}` = %s"); params.append(value)
#         status = 'completed'
#         update_sql = f"UPDATE `{USER_LISTING_TABLE_NAME}` SET {', '.join(set_clauses)}, processing_status = %s, last_processed_at = %s WHERE listId = %s"
#         params.extend([status, datetime.now(), listing_id])
#         cursor.execute(update_sql, tuple(params))
#         conn.commit()
#         print(f"TASK {task_id}: Processed and saved. Status: {status}.")
#     except Exception as e:
#         status = 'failed';
#         print(f"TASK {task_id}: UNHANDLED EXCEPTION: {e}");
#         traceback.print_exc()
#         if conn and conn.is_connected():
#             err_cur = None
#             try:
#                 err_cur = conn.cursor()
#                 err_cur.execute(
#                     f"UPDATE `{USER_LISTING_TABLE_NAME}` SET processing_status = %s, last_processed_at = %s WHERE listId = %s",
#                     (status, datetime.now(), listing_id))
#                 conn.commit()
#             except Exception as db_err_on_fail:
#                 print(f"TASK {task_id}: CRITICAL! Failed to update status to 'failed': {db_err_on_fail}")
#             finally:
#                 if err_cur: err_cur.close()
#     finally:
#         if cursor: cursor.close()
#         if conn and conn.is_connected(): conn.close()
#         if task_id in active_background_tasks:
#             try:
#                 del active_background_tasks[task_id]
#             except KeyError:
#                 pass
#         print(f"TASK {task_id}: Background processing FINISHED. Final Status: {status}.")
#         print_system_usage()


# --- Main Data Processing Task (Background) ---
async def process_data_and_save_all_to_db(listing_id: str, location_name_raw: str):
    market_table = get_market_table_name(location_name_raw)
    sanitized_loc = sanitize_location_name_for_table(location_name_raw)
    task_id = f"{CENTRAL_DB_NAME}:{sanitized_loc}:{listing_id}"
    print(f"TASK {task_id}: Background processing STARTED.")
    print_system_usage()
    conn, cursor, status = None, None, 'failed'  # Default status to failed

    DEFAULT_FALLBACK_BASE_PRICE = 100.00  # Define a sensible default for calendar if algo fails

    try:
        conn = get_central_db_connection()
        cursor = conn.cursor(dictionary=True)

        # Set status to pending in DB
        cursor.execute(
            f"UPDATE `{USER_LISTING_TABLE_NAME}` SET processing_status = %s, last_processed_at = %s WHERE listId = %s",
            ('pending', datetime.now(), listing_id)
        )
        if cursor.rowcount == 0:
            print(
                f"TASK {task_id}: Warning - Listing {listing_id} not found when setting to pending. This might be a new listing being inserted.")
            # If it's a new listing, initialize_listing_defaults_in_db should have inserted it.
            # If it's truly not there, the next select will fail.
        conn.commit()  # Commit the pending status
        status = 'pending'  # Update in-memory status

        # Fetch current listing data, including active_pricing_profile
        cursor.execute(f"SELECT *, active_pricing_profile FROM `{USER_LISTING_TABLE_NAME}` WHERE listId = %s",
                       (listing_id,))
        listing_dict = cursor.fetchone()
        if not listing_dict:
            print(f"TASK {task_id}: Listing {listing_id} vanished or was not found after setting to pending. Aborting.")
            # No need to set status to failed here, as it's already the default and task will end.
            return

        listing_df = pd.DataFrame([listing_dict])
        active_profile = listing_dict.get('active_pricing_profile') or "algorithmic"  # Default to algorithmic
        listing_lat = listing_df.get('lat', pd.Series([None])).iloc[0]
        listing_long = listing_df.get('long', pd.Series([None])).iloc[0]


        # --- Amenities Update ---
        try:
            central_cfg = {**BASE_DB_CONFIG, 'database': CENTRAL_DB_NAME}
            loop = asyncio.get_running_loop()
            await loop.run_in_executor(None, functools.partial(sql_update_all_amenities.process_amenities_in_db,
                                                               db_config_override=central_cfg,
                                                               table_name_override=USER_LISTING_TABLE_NAME))
        except Exception as amenities_err:
            print(f"TASK {task_id}: Amenities update error: {amenities_err}")

        # --- Market Data Fetch & Filter ---
        market_df = pd.DataFrame()
        cursor.execute(f"SHOW TABLES LIKE '{market_table}'")
        if cursor.fetchone():
            market_df = pd.read_sql_query(f"SELECT * FROM `{market_table}`", conn)
        else:
            print(f"TASK {task_id}: Market table '{market_table}' not found.")

        market_200_df = pd.DataFrame()
        if not market_df.empty and pd.notna(listing_lat) and pd.notna(listing_long):
            try:
                market_200_df = select_comparables_for_pricing(market_df, listing_df)
            except Exception as filter_err:
                print(f"TASK {task_id}: Market filter error: {filter_err}")

        results = {}  # To store JSON strings for DB update

        # --- Processed Amenities ---
        try:
            results["processed_amenities_json"] = json.dumps(
                timestamp_to_str(get_top_amenities(market_200_df, listing_df)))
        except Exception as e:
            results["processed_amenities_json"] = json.dumps({"error": f"Amenities: {e}"})

        # --- Min Stay Recommendations ---
        min_stay_profile = listing_dict.get('minStayProfile') or "default"
        try:
            min_stay_res = generate_recommendations(market_200_df, listing_df)
            min_stay_json_str = json.dumps(timestamp_to_str(min_stay_res))
            results["processed_min_stay_json"] = min_stay_json_str
            if min_stay_profile == "default":  # If user hasn't set a custom min_stay profile
                parsed_rec = json.loads(min_stay_json_str)
                # Only update customMinStayData if the recommendation is not an error
                if not (isinstance(parsed_rec, dict) and parsed_rec.get("error")):
                    results["customMinStayData"] = min_stay_json_str
            if not listing_dict.get('minStayProfile'):  # Ensure minStayProfile is set if it was NULL
                results["minStayProfile"] = "default"
        except Exception as e:
            err_json = json.dumps({"error": f"Min Stay: {e}"})
            results["processed_min_stay_json"] = err_json
            if min_stay_profile == "default":  # If default, store error in customMinStayData too
                results["customMinStayData"] = err_json
            if not listing_dict.get('minStayProfile'):
                results["minStayProfile"] = "default"

        # --- Algorithmic Pricing (processed_pricing_json) & Active Pricing Details Preparation ---
        raw_base_price_from_algo = None
        try:
            raw_base_price_from_algo = get_recommended_price(market_200_df, listing_df)
        except Exception as e:
            print(f"TASK {task_id}: Error calling get_recommended_price: {e}")
            raw_base_price_from_algo = None

        algo_pricing_details_dict = None  # This will be a Python dict
        algorithmic_base_price_is_valid = False

        if raw_base_price_from_algo is not None and pd.notna(raw_base_price_from_algo):
            try:
                bp = float(raw_base_price_from_algo)
                if bp > 0:
                    algo_pricing_details_dict = {
                        "base_price": round(bp, 2),
                        "max_price": round(bp * 2.0, 2),
                        "min_price": round(bp * 0.4, 2)
                    }
                    algorithmic_base_price_is_valid = True
                else:
                    algo_pricing_details_dict = {
                        "error": "Base Price: Algorithmic price not positive",
                        "base_price": None, "min_price": None, "max_price": None
                    }
            except (ValueError, TypeError):
                algo_pricing_details_dict = {
                    "error": "Base Price: Algorithmic price invalid format",
                    "base_price": None, "min_price": None, "max_price": None
                }
        else:
            algo_pricing_details_dict = {
                "error": "Base Price: Algorithm returned N/A",
                "base_price": None, "min_price": None, "max_price": None
            }

        results["processed_pricing_json"] = json.dumps(algo_pricing_details_dict)

        # Determine what goes into active_pricing_details for the DB update
        # This is critical for the subsequent calendar calculation on the *first run*
        if active_profile == "algorithmic":
            if algorithmic_base_price_is_valid:
                results["active_pricing_details"] = results[
                    "processed_pricing_json"]  # Use the valid algo price JSON string
            else:
                # Algorithmic pricing failed or invalid, but profile is "algorithmic".
                # We MUST provide a fallback for active_pricing_details for calendar generation.
                # processed_pricing_json will still show the actual error from the algorithm.
                print(
                    f"TASK {task_id}: Algorithmic base price failed or invalid. Using fallback {DEFAULT_FALLBACK_BASE_PRICE} for 'active_pricing_details' to enable initial calendar generation.")
                fallback_active_details_dict = {
                    "base_price": DEFAULT_FALLBACK_BASE_PRICE,
                    "min_price": round(DEFAULT_FALLBACK_BASE_PRICE * 0.4, 2),
                    "max_price": round(DEFAULT_FALLBACK_BASE_PRICE * 2.0, 2),
                    "note": "Fallback price used due to algorithmic pricing failure during initial processing."
                }
                results["active_pricing_details"] = json.dumps(fallback_active_details_dict)
        # If active_profile is NOT "algorithmic", active_pricing_details should have been set
        # by a previous call to /api/apply-pricing. If it's NULL or invalid for a non-algorithmic
        # profile, the calendar might still fail, but that's a different data integrity issue.
        # The initialize_listing_defaults_in_db should set active_pricing_profile to 'algorithmic'
        # if it's NULL, and also sets default occupancy profiles.

        if not listing_dict.get(
                'active_pricing_profile'):  # Ensure active_pricing_profile is set in results if it was NULL
            results["active_pricing_profile"] = "algorithmic"

        # --- Other Processed Data ---
        try:
            results["processed_market_pricing_json"] = json.dumps(
                timestamp_to_str(generate_market_pricing(market_200_df)))
        except Exception as e:
            results["processed_market_pricing_json"] = json.dumps({"error": f"Market Tiers: {e}"})

        try:
            results["processed_monthly_averages_json"] = json.dumps(
                timestamp_to_str(calculate_monthly_averages(listing_df, market_200_df)))
        except Exception as e:
            results["processed_monthly_averages_json"] = json.dumps({"error": f"Monthly Avg: {e}"})

        try:
            map_cols = ['price', 'lat', 'long', 'color']  # Ensure 'color' column exists if used
            # Check if market_200_df has these columns before trying to select them
            valid_map_cols = [col for col in map_cols if col in market_200_df.columns]
            if not market_200_df.empty and valid_map_cols == map_cols:  # Or adjust condition as needed
                results["processed_map_recs_json"] = json.dumps(
                    timestamp_to_str(market_200_df[valid_map_cols].dropna().to_dict('records')))
            else:
                missing_cols = [col for col in map_cols if col not in market_200_df.columns]
                if missing_cols:
                    print(f"TASK {task_id}: Map recs missing columns: {missing_cols}")
                results["processed_map_recs_json"] = json.dumps([])  # Empty list if data is insufficient
        except Exception as e:
            results["processed_map_recs_json"] = json.dumps({"error": f"Map Recs: {e}"})

        # --- Update Database with all processed results ---
        # `active_pricing_details` is now prepared in `results`
        set_clauses, params_list = [], []
        for key, value in results.items():
            set_clauses.append(f"`{key}` = %s")
            params_list.append(value)

        # The status here will be 'pending' from the earlier commit.
        # We will update it to 'completed' or 'failed' at the very end.
        # For now, just update the data columns.
        if set_clauses:  # Only run update if there's something to update
            update_data_sql = f"UPDATE `{USER_LISTING_TABLE_NAME}` SET {', '.join(set_clauses)} WHERE listId = %s"
            cursor.execute(update_data_sql, tuple(params_list + [listing_id]))
        # No commit here yet, do it after dependent calculations or at the end of try block.

        # --- Calculate and Update Dependent Data (Daily Adjustments, Calendar Pricing) ---
        # These functions will now read the `active_pricing_details` we just prepared/updated.
        try:
            _calculate_and_update_listing_daily_adjustments_in_db(listing_id, conn, cursor_to_use=cursor)
        except Exception as e:
            print(f"TASK {task_id}: Daily adjustment step error: {e}")
            # Decide if this error should make the overall status 'failed' or 'partially_completed'

        try:
            _calculate_and_update_calendar_pricing_in_db(listing_id, conn, cursor_to_use=cursor)
        except Exception as e:
            print(f"TASK {task_id}: ERROR during calendar pricing step: {e}")
            # This error is the one we are trying to avoid by preparing active_pricing_details

        # All operations successful up to this point (or errors handled within sub-functions)
        status = 'completed'  # Mark as completed
        conn.commit()  # Commit all changes: main data, daily adjustments, calendar pricing
        print(f"TASK {task_id}: Processed and saved. Status: {status}.")

    except Exception as e:
        status = 'failed'  # Ensure status is 'failed' on any unhandled exception in the main try block
        print(f"TASK {task_id}: UNHANDLED EXCEPTION during processing: {e}")
        traceback.print_exc()
        if conn and conn.is_connected():
            try:
                conn.rollback()  # Rollback any uncommitted changes if an error occurred mid-transaction
            except Exception as rb_err:
                print(f"TASK {task_id}: Error during rollback: {rb_err}")
    finally:
        # Always update the final status in the database, regardless of success or failure
        if conn and conn.is_connected() and cursor:  # Check if cursor is valid
            try:
                # Fetch current status to avoid overwriting if another process changed it, though unlikely here
                # For simplicity, just update with the determined status
                cursor.execute(
                    f"UPDATE `{USER_LISTING_TABLE_NAME}` SET processing_status = %s, last_processed_at = %s WHERE listId = %s",
                    (status, datetime.now(), listing_id)
                )
                conn.commit()
            except Exception as db_err_on_final_status_update:
                print(
                    f"TASK {task_id}: CRITICAL! Failed to update final status to '{status}': {db_err_on_final_status_update}")

        if cursor:
            try:
                cursor.close()
            except Exception as cur_close_err:
                print(f"TASK {task_id}: Error closing cursor: {cur_close_err}")
        if conn and conn.is_connected():
            try:
                conn.close()
            except Exception as conn_close_err:
                print(f"TASK {task_id}: Error closing connection: {conn_close_err}")

        if task_id in active_background_tasks:
            try:
                del active_background_tasks[task_id]
            except KeyError:
                pass  # Task might have been removed by another mechanism or already finished
        print(f"TASK {task_id}: Background processing FINISHED. Final Status: {status}.")
        print_system_usage()


def _recalculate_all_specific_date_adjustments(
        listing_id: str,
        new_reference_base_price: float,
        db_conn: mysql.connector.MySQLConnection,
        cursor_to_use: mysql.connector.cursor.MySQLCursorDict  # Use MySQLCursorDict for dictionary=True
):
    """
    Recalculates all existing specific date percentage adjustments based on a new reference base price.
    It reads the stored 'mkt_price' for each date and computes a new 'adj_perc'.
    """
    if not isinstance(new_reference_base_price, (int, float)) or new_reference_base_price <= 0:
        print(
            f"RECALC_SPECIFIC ({listing_id}): Invalid new_reference_base_price ({new_reference_base_price}). Skipping recalculation.")
        return

    try:
        cursor_to_use.execute(
            f"SELECT specific_date_pricing_json FROM `{USER_LISTING_TABLE_NAME}` WHERE listId = %s", (listing_id,)
        )
        result = cursor_to_use.fetchone()
    except MySQLError as e:
        print(f"RECALC_SPECIFIC ({listing_id}): DB error fetching specific_date_pricing_json: {e}")
        return  # Or raise to allow transaction rollback

    if not result or not result.get('specific_date_pricing_json'):
        print(f"RECALC_SPECIFIC ({listing_id}): No existing specific_date_pricing_json to recalculate.")
        return

    try:
        current_specific_adjustments = json.loads(result['specific_date_pricing_json'])
        if not isinstance(current_specific_adjustments, dict):
            print(f"RECALC_SPECIFIC ({listing_id}): Malformed specific_date_pricing_json (not a dict). Skipping.")
            return
    except json.JSONDecodeError:
        print(f"RECALC_SPECIFIC ({listing_id}): Error decoding specific_date_pricing_json. Skipping.")
        return

    updated_adjustments: Dict[str, Dict[str, float]] = {}
    recalculation_performed_on_at_least_one_entry = False

    for date_str, entry_data in current_specific_adjustments.items():
        if isinstance(entry_data, dict) and "mkt_price" in entry_data and "adj_perc" in entry_data:
            try:
                market_price_for_date = float(entry_data["mkt_price"])
                if market_price_for_date <= 0:  # Market price should be positive
                    print(
                        f"RECALC_SPECIFIC ({listing_id}): Invalid mkt_price ({market_price_for_date}) for {date_str}. Keeping old adj_perc.")
                    updated_adjustments[date_str] = entry_data  # Keep original if mkt_price is bad
                    continue

                # Calculate new base percentage difference for this specific date's market price
                # against the new overall reference base price
                new_base_perc_diff = ((
                                                  market_price_for_date - new_reference_base_price) / new_reference_base_price) * 100

                # Apply random variation to this new base percentage
                recalculated_adj_perc = apply_percentage_variation(new_base_perc_diff)

                updated_adjustments[date_str] = {
                    "adj_perc": round(recalculated_adj_perc, 2),
                    "mkt_price": round(market_price_for_date, 2)  # mkt_price itself doesn't change here
                }
                recalculation_performed_on_at_least_one_entry = True
            except (ValueError, TypeError) as e:
                print(f"RECALC_SPECIFIC ({listing_id}): Error processing entry for {date_str}: {e}. Keeping old entry.")
                updated_adjustments[date_str] = entry_data  # Keep original if error during processing
        else:
            # Entry is not in the expected new format.
            # Decide how to handle: skip, try to convert, or log and carry over.
            # For now, log and carry over to avoid data loss, but this indicates a data integrity issue.
            print(
                f"RECALC_SPECIFIC ({listing_id}): Entry for {date_str} is not in the expected format. Carrying over original entry.")
            updated_adjustments[date_str] = entry_data

    if recalculation_performed_on_at_least_one_entry:
        try:
            updated_json = json.dumps(updated_adjustments)
            cursor_to_use.execute(
                f"UPDATE `{USER_LISTING_TABLE_NAME}` SET specific_date_pricing_json = %s WHERE listId = %s",
                (updated_json, listing_id)
            )
            print(f"RECALC_SPECIFIC ({listing_id}): Successfully recalculated and updated specific_date_pricing_json.")
        except MySQLError as e:
            print(
                f"RECALC_SPECIFIC ({listing_id}): DB error updating with recalculated specific_date_pricing_json: {e}")
            # Consider raising e to allow transaction rollback by the caller
        except Exception as e:
            print(f"RECALC_SPECIFIC ({listing_id}): Unexpected error during DB update: {e}")
            # Consider raising e
    else:
        print(f"RECALC_SPECIFIC ({listing_id}): No valid entries found or no changes made during recalculation.")

# --- API Endpoints ---
@app.post("/api/process-listing", status_code=202, summary="Initiate Listing Data Processing")
async def process_listing_data_endpoint(request: ListingProcessRequest):
    loc_name, list_id = request.location_name, request.listing_id
    market_table = get_market_table_name(loc_name)
    task_key = f"{CENTRAL_DB_NAME}:{sanitize_location_name_for_table(loc_name)}:{list_id}"
    if task_key in active_background_tasks and not active_background_tasks[task_key].done():
        raise HTTPException(status_code=409, detail=f"Processing already in progress for {list_id}.")
    if task_key in active_background_tasks: del active_background_tasks[task_key]
    conn = None
    try:
        conn = get_central_db_connection()
        ensure_table_exists(conn, USER_LISTING_TABLE_NAME, USER_TABLE_CREATE_SQL)
        ensure_table_exists(conn, market_table, MARKET_TABLE_CREATE_SQL)
        ensure_required_columns_exist(conn)
        initialize_listing_defaults_in_db(conn, list_id)
        conn.commit()
        bg_task = asyncio.create_task(process_data_and_save_all_to_db(list_id, loc_name))
        active_background_tasks[task_key] = bg_task
        return {"message": "Data processing initiated.", "listing_id": list_id, "task_key": task_key}
    except HTTPException as http_exc:
        if conn: conn.rollback(); raise http_exc
    except Exception as e:
        if conn: conn.rollback()
        print(f"API: CRITICAL Error initiating processing for {list_id}/{loc_name}: {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Failed to initiate processing: {str(e)}")
    finally:
        if conn and conn.is_connected(): conn.close()


@app.get("/api/active-pricing-profile", summary="Get Active Pricing Profile and Details",
         response_model=GetActivePricingProfileResponse)
async def get_active_pricing_profile_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    table = USER_LISTING_TABLE_NAME;
    conn, cur = None, None
    try:
        conn = get_central_db_connection();
        cur = conn.cursor(dictionary=True)
        cur.execute(
            f"SELECT listId, active_pricing_profile, active_pricing_details, processed_pricing_json FROM `{table}` WHERE listId = %s",
            (listing_id,))
        data = cur.fetchone()
        active_prof, active_details = "algorithmic", None
        if not data:
            cur.execute(f"SHOW TABLES LIKE '{table}'")
            if not cur.fetchone(): raise HTTPException(status_code=404, detail=f"Table '{table}' not found.")
            return GetActivePricingProfileResponse(listing_id=listing_id, location_name=location_name,
                                                   active_pricing_profile=active_prof, active_pricing_details=None)
        active_prof = data.get("active_pricing_profile") or "algorithmic"
        active_details_json = data.get("active_pricing_details")
        if active_details_json:
            try:
                active_details = json.loads(active_details_json)
            except json.JSONDecodeError:
                active_details = None
        if not active_details:
            if active_prof == "algorithmic":
                processed_str = data.get("processed_pricing_json")
                if processed_str:
                    try:
                        active_details = json.loads(processed_str)
                        if not all(k in active_details for k in
                                   ["base_price", "min_price", "max_price"]): active_details = None
                    except json.JSONDecodeError:
                        active_details = None
            else:
                active_details = {"base_price": None, "min_price": None, "max_price": None}
        return GetActivePricingProfileResponse(listing_id=str(data["listId"]), location_name=location_name,
                                               active_pricing_profile=active_prof,
                                               active_pricing_details=active_details)
    except MySQLError as e:
        print(f"API: DB Error active pricing {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
            status_code=500, detail=f"DB error: {str(e)}")
    except Exception as e:
        print(f"API: General Error active pricing {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
            status_code=500, detail=f"Internal error: {str(e)}")
    finally:
        if cur: cur.close()
        if conn and conn.is_connected(): conn.close()


# @app.post("/api/apply-pricing", summary="Apply and Save Chosen Pricing Strategy and Details")
# async def apply_pricing_strategy_endpoint(payload: AppliedPricingStrategy):
#     table = USER_LISTING_TABLE_NAME;
#     list_id = payload.listing_id
#     conn, cur = None, None
#     try:
#         conn = get_central_db_connection();
#         cur = conn.cursor(dictionary=True)
#         active_details_data = {"base_price": payload.base_price, "min_price": payload.min_price,
#                                "max_price": payload.max_price}
#
#         market_segment_value = None
#         if payload.pricing_strategy.startswith("market-"):
#             parts = payload.pricing_strategy.split("-", 1)
#             if len(parts) > 1:
#                 market_segment_value = parts[1]
#         if market_segment_value:
#             active_details_data["market_segment"] = market_segment_value
#
#         active_details_json = json.dumps(active_details_data)
#         update_data = {"active_pricing_profile": payload.pricing_strategy,
#                        "active_pricing_details": active_details_json, "last_processed_at": datetime.now()}
#         set_clauses = ", ".join([f"`{key}` = %s" for key in update_data.keys()])
#         sql_params = list(update_data.values()) + [list_id]
#         cur.execute(f"UPDATE `{table}` SET {set_clauses} WHERE listId = %s", tuple(sql_params))
#         if cur.rowcount == 0:
#             conn.rollback()
#             cur.execute(f"SHOW TABLES LIKE '{table}'")
#             if not cur.fetchone():
#                 raise HTTPException(status_code=404, detail=f"Table '{table}' not found.")
#             else:
#                 raise HTTPException(status_code=404, detail=f"Listing '{list_id}' not found.")
#         _calculate_and_update_calendar_pricing_in_db(list_id, conn, cursor_to_use=cur)
#         conn.commit()
#         return {
#             "message": f"Pricing strategy '{payload.pricing_strategy}' applied and calendar pricing updated for {list_id}."}
#     except MySQLError as e:
#         if conn: conn.rollback(); print(
#             f"API: DB Error apply pricing {list_id}: {e}"); traceback.print_exc(); raise HTTPException(status_code=500,
#                                                                                                        detail=f"DB error: {str(e)}")
#     except HTTPException as http_exc:
#         if conn: conn.rollback(); raise http_exc
#     except Exception as e:
#         if conn: conn.rollback(); print(
#             f"API: General Error apply pricing {list_id}: {e}"); traceback.print_exc(); raise HTTPException(
#             status_code=500, detail=f"Internal error: {str(e)}")
#     finally:
#         if cur: cur.close()
#         if conn and conn.is_connected(): conn.close()


@app.post("/api/apply-pricing", summary="Apply and Save Chosen Pricing Strategy and Details")
async def apply_pricing_strategy_endpoint(payload: AppliedPricingStrategy):
    table = USER_LISTING_TABLE_NAME
    list_id = payload.listing_id
    conn, cur = None, None
    try:
        conn = get_central_db_connection()
        cur = conn.cursor(dictionary=True)  # Ensure cursor is dictionary type

        new_active_base_price = payload.base_price  # This is the new reference base price

        active_details_data = {
            "base_price": new_active_base_price,
            "min_price": payload.min_price,
            "max_price": payload.max_price
        }

        market_segment_value = None
        if payload.pricing_strategy.startswith("market-"):
            parts = payload.pricing_strategy.split("-", 1)
            if len(parts) > 1:
                market_segment_value = parts[1]
        if market_segment_value:
            active_details_data["market_segment"] = market_segment_value

        active_details_json = json.dumps(active_details_data)

        update_data_dict = {
            "active_pricing_profile": payload.pricing_strategy,
            "active_pricing_details": active_details_json,
            "last_processed_at": datetime.now()
        }

        set_clauses = ", ".join([f"`{key}` = %s" for key in update_data_dict.keys()])
        sql_params = list(update_data_dict.values()) + [list_id]

        cur.execute(f"UPDATE `{table}` SET {set_clauses} WHERE listId = %s", tuple(sql_params))

        if cur.rowcount == 0:
            conn.rollback()  # Rollback before checking table existence
            cur.execute(f"SHOW TABLES LIKE '{table}'")  # Check if table exists
            if not cur.fetchone():
                raise HTTPException(status_code=404, detail=f"Table '{table}' not found.")
            else:
                raise HTTPException(status_code=404, detail=f"Listing '{list_id}' not found.")

        # --- RECALCULATE SPECIFIC DATE ADJUSTMENTS ---
        if new_active_base_price > 0:
            _recalculate_all_specific_date_adjustments(list_id, new_active_base_price, conn, cur)
        else:
            print(
                f"APPLY_PRICING ({list_id}): New active base price is not positive ({new_active_base_price}). Skipping specific date recalculation.")
        # --- END RECALCULATION ---

        _calculate_and_update_calendar_pricing_in_db(list_id, conn, cursor_to_use=cur)

        conn.commit()
        return {
            "message": f"Pricing strategy '{payload.pricing_strategy}' applied. Specific date adjustments (if any) recalculated. Calendar pricing updated for {list_id}."
        }
    except MySQLError as e:
        if conn: conn.rollback()
        print(f"API: DB Error apply pricing {list_id}: {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"DB error: {str(e)}")
    except HTTPException as http_exc:
        if conn: conn.rollback()  # Ensure rollback for HTTPExceptions raised within try block
        raise http_exc
    except Exception as e:
        if conn: conn.rollback()
        print(f"API: General Error apply pricing {list_id}: {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Internal error: {str(e)}")
    finally:
        if cur: cur.close()
        if conn and conn.is_connected(): conn.close()

@app.get("/api/listing-status", summary="Get Listing Processing Status")
async def get_listing_status_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    task_key = f"{CENTRAL_DB_NAME}:{sanitize_location_name_for_table(location_name)}:{listing_id}"
    conn, cur = None, None
    if task_key in active_background_tasks and not active_background_tasks[task_key].done():
        return {"listing_id": listing_id, "processing_status": "pending", "source": "active_task_list"}
    try:
        conn = get_central_db_connection();
        cur = conn.cursor(dictionary=True)
        cur.execute(f"SELECT processing_status, last_processed_at FROM `{USER_LISTING_TABLE_NAME}` WHERE listId = %s",
                    (listing_id,))
        status_data = cur.fetchone()
        if not status_data:
            cur.execute(f"SHOW TABLES LIKE '{USER_LISTING_TABLE_NAME}'")
            if not cur.fetchone():
                raise HTTPException(status_code=404, detail=f"Table '{USER_LISTING_TABLE_NAME}' not found.")
            else:
                raise HTTPException(status_code=404, detail=f"Listing '{listing_id}' not found.")
        return {"listing_id": listing_id, "processing_status": status_data.get('processing_status'),
                "last_processed_at": timestamp_to_str(status_data.get('last_processed_at')), "source": "database"}
    except MySQLError as e:
        print(f"API: DB Error status {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(status_code=500,
                                                                                                     detail="DB error.")
    except HTTPException as http_exc:
        raise http_exc
    except Exception as e:
        print(f"API: Unexpected error status {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
            status_code=500, detail="Internal error.")
    finally:
        if cur: cur.close()
        if conn and conn.is_connected(): conn.close()


async def get_processed_data_from_db(listing_id: str, location_name_raw: str, column_name: str,
                                     error_prefix: str) -> Any:
    table = USER_LISTING_TABLE_NAME;
    conn, cur = None, None
    try:
        conn = get_central_db_connection();
        cur = conn.cursor(dictionary=True)
        cur.execute(f"SELECT `{column_name}`, processing_status FROM `{table}` WHERE listId = %s", (listing_id,))
        result = cur.fetchone()
        if not result:
            cur.execute(f"SHOW TABLES LIKE '{table}'")
            if not cur.fetchone():
                raise HTTPException(status_code=404, detail=f"{error_prefix}: Table '{table}' not found.")
            else:
                raise HTTPException(status_code=404, detail=f"{error_prefix}: Listing '{listing_id}' not found.")
        json_str, status = result.get(column_name), result.get('processing_status')
        if not json_str:  # Handles NULL or empty string
            if status == 'pending': raise HTTPException(status_code=202,
                                                        detail=f"{error_prefix} for '{listing_id}' is processing.")
            if status == 'failed': raise HTTPException(status_code=500,
                                                       detail=f"{error_prefix} for '{listing_id}' failed processing.")
            # For specific_date_pricing_json, NULL means no specific adjustments yet, return empty dict
            return {} if column_name == "specific_date_pricing_json" else {}  # Or [] if list expected
        try:
            parsed = json.loads(json_str)
            if isinstance(parsed, dict) and "error" in parsed and parsed["error"]:
                raise HTTPException(status_code=500,
                                    detail=f"Error retrieving {error_prefix.lower()}: Calculation failed ({parsed['error']})")
            return parsed
        except json.JSONDecodeError:
            raise HTTPException(status_code=500, detail=f"Invalid data format for {error_prefix.lower()}. Reprocess.")
        except HTTPException as http_exc:
            raise http_exc
        except Exception as e:
            print(
                f"API: Unexpected error processing {column_name} for {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
                status_code=500, detail=f"Internal error retrieving {error_prefix.lower()}.")
    except MySQLError as e:
        if e.errno == MySQLError.ER_BAD_FIELD_ERROR: raise HTTPException(status_code=500,
                                                                         detail=f"{error_prefix}: Column '{column_name}' not found. Schema outdated or processing failed.")
        print(f"API: DB Error get data ({column_name}) for {listing_id}: {e}");
        traceback.print_exc();
        raise HTTPException(status_code=500, detail="DB error.")
    except HTTPException as http_exc:
        raise http_exc
    except Exception as e:
        print(
            f"API: Unexpected error in get_processed_data_from_db for {column_name}/{listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
            status_code=500, detail="Internal error.")
    finally:
        if cur: cur.close()
        if conn and conn.is_connected(): conn.close()


@app.get("/api/amenities", summary="Get Processed Amenities Data")
async def get_amenities_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "processed_amenities_json", "Amenities data")


@app.get("/api/pricing", summary="Get Recommended Base Pricing")
async def get_pricing_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "processed_pricing_json", "Pricing data")


@app.get("/api/min-stay-rec", summary="Get Minimum Stay Recommendations")
async def get_min_stay_rec_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "processed_min_stay_json",
                                            "Min stay recommendations")


@app.get("/api/map-recs", summary="Get Map Recommendations")
async def get_map_recs_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "processed_map_recs_json", "Map recommendations")


@app.get("/api/monthly-averages", summary="Get Monthly Averages (Property vs Market)")
async def get_monthly_averages_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "processed_monthly_averages_json",
                                            "Monthly averages")


@app.get("/api/market-pricing", summary="Get Market Pricing Tiers")
async def get_market_pricing_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "processed_market_pricing_json",
                                            "Market pricing tiers")


@app.get("/api/daily-occupancy-adjustments", summary="Get Daily Occupancy Price Adjustments")
async def get_daily_occupancy_adjustments_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "daily_occupancy_adjustments_json",
                                            "Daily occupancy adjustments")


@app.get("/api/calendar-pricing", summary="Get 1-Year Calendar Pricing")
async def get_calendar_pricing_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    return await get_processed_data_from_db(listing_id, location_name, "calendar_pricing_json", "Calendar pricing data")


@app.get("/api/customization", summary="Get Listing Customization Settings")
async def get_customization_settings_endpoint(listing_id: str = Query(...), location_name: str = Query(...)):
    table = USER_LISTING_TABLE_NAME;
    conn, cur = None, None
    try:
        conn = get_central_db_connection();
        cur = conn.cursor(dictionary=True)
        cur.execute(
            f"SELECT listId, occupancyEnabled, occupancyProfile, occupancyProfileDetails, minStayEnabled, minStayProfile, customMinStayData FROM `{table}` WHERE listId = %s",
            (listing_id,))
        settings = cur.fetchone()
        if not settings:
            cur.execute(f"SHOW TABLES LIKE '{table}'")
            if not cur.fetchone():
                raise HTTPException(status_code=404, detail=f"Table '{table}' not found.")
            else:
                raise HTTPException(status_code=404, detail=f"Settings not found for listing '{listing_id}'.")
        profile_details, min_stay_details = {}, {}
        try:
            prof_name = settings.get('occupancyProfile') or 'default';
            details_str = settings.get('occupancyProfileDetails')
            if details_str:
                loaded = json.loads(details_str)
                if isinstance(loaded, dict) and loaded:
                    profile_details = loaded
                else:
                    profile_details = ALL_PREDEFINED_PRICING_PROFILES.get(prof_name,
                                                                          ALL_PREDEFINED_PRICING_PROFILES['default'])
            else:
                profile_details = ALL_PREDEFINED_PRICING_PROFILES.get(prof_name,
                                                                      ALL_PREDEFINED_PRICING_PROFILES['default'])
        except (TypeError, json.JSONDecodeError):
            profile_details = ALL_PREDEFINED_PRICING_PROFILES.get(settings.get('occupancyProfile') or 'default',
                                                                  ALL_PREDEFINED_PRICING_PROFILES['default'])
        try:
            min_stay_str = settings.get('customMinStayData')
            if min_stay_str:
                loaded_min = json.loads(min_stay_str)
                if isinstance(loaded_min, dict): min_stay_details = loaded_min
        except (TypeError, json.JSONDecodeError):
            min_stay_details = {}
        return {"listing_id": settings["listId"], "occupancyEnabled": settings.get("occupancyEnabled", True),
                "occupancyProfile": settings.get("occupancyProfile") or "default", "customPricingData": profile_details,
                "minStayEnabled": settings.get("minStayEnabled", True),
                "minStayProfile": settings.get("minStayProfile") or "default", "customMinStayData": min_stay_details}
    except MySQLError as e:
        print(f"API: DB Error get customization {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
            status_code=500, detail="DB error.")
    except HTTPException as http_exc:
        raise http_exc
    except Exception as e:
        print(f"API: Unexpected error get customization {listing_id}: {e}"); traceback.print_exc(); raise HTTPException(
            status_code=500, detail="Internal error.")
    finally:
        if cur: cur.close()
        if conn and conn.is_connected(): conn.close()


# @app.post("/api/customization", summary="Update Listing Customization Settings")
# async def update_customization_settings_endpoint(payload: CustomizationUpdateRequest):
#     table, list_id = USER_LISTING_TABLE_NAME, payload.listing_id
#     conn, cur = None, None
#     try:
#         conn = get_central_db_connection();
#         cur = conn.cursor(dictionary=True)
#         prof_name, details_json = payload.occupancyProfile, None
#         if prof_name == "custom":
#             if not isinstance(payload.customPricingData, dict) or not payload.customPricingData: raise HTTPException(
#                 status_code=400, detail="Invalid 'customPricingData'.")
#             details_json = json.dumps(payload.customPricingData)
#         elif prof_name in ALL_PREDEFINED_PRICING_PROFILES:
#             details_json = json.dumps(ALL_PREDEFINED_PRICING_PROFILES[prof_name])
#         else:
#             prof_name = "default"; details_json = json.dumps(ALL_PREDEFINED_PRICING_PROFILES["default"])
#         min_stay_json = json.dumps(payload.customMinStayData) if isinstance(payload.customMinStayData,
#                                                                             dict) else json.dumps({})
#         update_params = (
#         payload.occupancyEnabled, prof_name, details_json, payload.minStayEnabled, payload.minStayProfile,
#         min_stay_json, list_id)
#         cur.execute(
#             f"UPDATE `{table}` SET occupancyEnabled=%s, occupancyProfile=%s, occupancyProfileDetails=%s, minStayEnabled=%s, minStayProfile=%s, customMinStayData=%s WHERE listId=%s",
#             update_params)
#         if cur.rowcount == 0:
#             conn.rollback()
#             cur.execute(f"SHOW TABLES LIKE '{table}'")
#             if not cur.fetchone():
#                 raise HTTPException(status_code=404, detail=f"Table '{table}' not found.")
#             else:
#                 raise HTTPException(status_code=404, detail=f"Listing '{list_id}' not found.")
#         _calculate_and_update_listing_daily_adjustments_in_db(list_id, conn, cursor_to_use=cur)
#         conn.commit()
#         return {"message": f"Customization settings for {list_id} updated."}
#     except MySQLError as e:
#         if conn: conn.rollback(); print(
#             f"API: DB Error update customization {list_id}: {e}"); traceback.print_exc(); raise HTTPException(
#             status_code=500, detail=f"DB error: {str(e)}")
#     except HTTPException as http_exc:
#         if conn: conn.rollback(); raise http_exc
#     except Exception as e:
#         if conn: conn.rollback(); print(
#             f"API: General Error update customization {list_id}: {e}"); traceback.print_exc(); raise HTTPException(
#             status_code=500, detail=f"Internal error: {str(e)}")
#     finally:
#         if cur: cur.close()
#         if conn and conn.is_connected(): conn.close()


@app.post("/api/customization", summary="Update Listing Customization Settings")
async def update_customization_settings_endpoint(payload: CustomizationUpdateRequest):
    table = USER_LISTING_TABLE_NAME  # Defined globally or as a constant
    list_id = payload.listing_id
    conn = None
    cur = None  # Initialize cur to None

    try:
        conn = get_central_db_connection()
        cur = conn.cursor(dictionary=True)  # Ensure cursor is dictionary type

        # Determine occupancyProfileDetails based on occupancyProfile
        prof_name = payload.occupancyProfile
        details_json_str: Optional[str] = None

        if prof_name == "custom":
            if not isinstance(payload.customPricingData, dict) or not payload.customPricingData:
                raise HTTPException(
                    status_code=400,
                    detail="Invalid 'customPricingData'. Must be a non-empty dictionary when occupancyProfile is 'custom'."
                )
            details_json_str = json.dumps(payload.customPricingData)
        elif prof_name in ALL_PREDEFINED_PRICING_PROFILES:
            details_json_str = json.dumps(ALL_PREDEFINED_PRICING_PROFILES[prof_name])
        else:
            # Default to "default" profile if an unrecognized profile name is given
            print(
                f"Warning: Unrecognized occupancyProfile '{prof_name}' for listing {list_id}. Defaulting to 'default'.")
            prof_name = "default"
            details_json_str = json.dumps(ALL_PREDEFINED_PRICING_PROFILES["default"])

        # Prepare minStayData
        min_stay_json_str = json.dumps(payload.customMinStayData) if isinstance(payload.customMinStayData,
                                                                                dict) else json.dumps({})

        # Prepare update parameters for the SQL query
        update_values = {
            "occupancyEnabled": payload.occupancyEnabled,
            "occupancyProfile": prof_name,
            "occupancyProfileDetails": details_json_str,
            "minStayEnabled": payload.minStayEnabled,
            "minStayProfile": payload.minStayProfile,
            "customMinStayData": min_stay_json_str,
            "last_processed_at": datetime.now()  # Also update last_processed_at or a new last_customized_at
        }

        set_clauses = ", ".join([f"`{key}` = %s" for key in update_values.keys()])
        sql_params = list(update_values.values()) + [list_id]

        cur.execute(
            f"UPDATE `{table}` SET {set_clauses} WHERE listId = %s",
            tuple(sql_params)
        )

        if cur.rowcount == 0:
            conn.rollback()  # Rollback before checking table existence
            # Check if the table itself exists if no rows were updated
            cur.execute(f"SHOW TABLES LIKE '{table}'")
            if not cur.fetchone():
                raise HTTPException(status_code=404, detail=f"Table '{table}' not found.")
            else:
                # Table exists, so listing_id must not exist
                raise HTTPException(status_code=404, detail=f"Listing '{list_id}' not found for update.")

        # 1. Recalculate daily occupancy adjustments based on the new settings
        print(f"CUSTOMIZATION_UPDATE ({list_id}): Recalculating daily occupancy adjustments.")
        _calculate_and_update_listing_daily_adjustments_in_db(list_id, conn, cursor_to_use=cur)

        # 2. Recalculate calendar pricing to reflect new daily occupancy adjustments
        print(f"CUSTOMIZATION_UPDATE ({list_id}): Recalculating calendar pricing due to customization changes.")
        _calculate_and_update_calendar_pricing_in_db(list_id, conn, cursor_to_use=cur)

        conn.commit()
        return {
            "message": f"Customization settings for listing '{list_id}' updated. Daily occupancy adjustments and calendar pricing have been recalculated."
        }

    except MySQLError as e:
        if conn:
            conn.rollback()
        print(f"API (/api/customization): DB Error for listing {list_id}: {e}")
        traceback.print_exc()  # Make sure traceback is imported: import traceback
        raise HTTPException(status_code=500, detail=f"Database error occurred: {str(e)}")
    except HTTPException as http_exc:
        # This will catch HTTPExceptions raised explicitly (e.g., for bad customPricingData)
        if conn:
            conn.rollback()
        raise http_exc  # Re-raise the HTTPException
    except Exception as e:
        if conn:
            conn.rollback()
        print(f"API (/api/customization): General Error for listing {list_id}: {e}")
        traceback.print_exc()  # Make sure traceback is imported
        raise HTTPException(status_code=500, detail=f"An internal server error occurred: {str(e)}")
    finally:
        if cur:
            cur.close()
        if conn and conn.is_connected():
            conn.close()

@app.get("/api/global-status", summary="Get Global Processing Status Overview")
async def get_global_status():
    active_info, to_remove = {}, []
    for key, task in active_background_tasks.items():
        if task and not task.done():
            active_info[key] = "running"
        else:
            to_remove.append(key)
    for key in to_remove:
        try:
            del active_background_tasks[key]
        except KeyError:
            pass
    return {"active_processing_tasks_count": len(active_info), "active_tasks_by_key": active_info}


def get_date_range_market_table_name(loc_name_raw: str, start_dt_obj: date, end_dt_obj: date) -> str:
    sanitized_loc = sanitize_location_name_for_table(loc_name_raw)
    start_str, end_str = start_dt_obj.strftime("%d%b").lower(), end_dt_obj.strftime("%d%b").lower()
    year_str = str(start_dt_obj.year)
    return f"{start_str}_{end_str}-{year_str}_{sanitized_loc}"


# --- NEW/MODIFIED ENDPOINTS for Specific Date Pricing ---

# @app.post("/api/market-specific-date-range-pricing", summary="Calculate and Store Date-Specific Percentage Adjustments",
#           response_model=SpecificDatePricingResponse)
# async def market_specific_date_range_pricing_endpoint(payload: SpecificDatePricingRequest = Body(...)):
#     conn, cursor = None, None
#     listing_id = payload.listing_id
#     location_name = payload.location_name
#     start_date_str = payload.start_date
#     end_date_str = payload.end_date
#
#     try:
#         start_date_obj = datetime.strptime(start_date_str, "%Y-%m-%d").date()
#         end_date_obj = datetime.strptime(end_date_str, "%Y-%m-%d").date()
#         if start_date_obj > end_date_obj: raise ValueError("Start date cannot be after end date.")
#     except ValueError as e:
#         raise HTTPException(status_code=400, detail=f"Invalid date format or range: {e}. Use YYYY-MM-DD.")
#
#     market_table_for_range = get_date_range_market_table_name(location_name, start_date_obj, end_date_obj)
#
#     processed_base_price: Optional[float] = None
#     market_range_base_price: Optional[float] = None
#     base_percentage_diff: Optional[float] = None
#     daily_percentage_adjustments: Dict[str, float] = {}
#     message = "Processing started."
#
#     try:
#         conn = get_central_db_connection()
#         cursor = conn.cursor(dictionary=True)
#
#         # 1. Fetch user listing data (processed_pricing_json, lat, long, specific_date_pricing_json)
#         cursor.execute(
#             f"SELECT listId, lat, `long`, processed_pricing_json, specific_date_pricing_json FROM `{USER_LISTING_TABLE_NAME}` WHERE listId = %s",
#             (listing_id,)
#         )
#         user_data = cursor.fetchone()
#         if not user_data:
#             raise HTTPException(status_code=404, detail=f"User listing '{listing_id}' not found.")
#
#         user_listing_df = pd.DataFrame([user_data])  # For get_recommended_price
#         user_lat, user_long = user_data.get('lat'), user_data.get('long')
#
#         # Parse processed_pricing_json to get P_processed
#         processed_pricing_str = user_data.get('processed_pricing_json')
#         if not processed_pricing_str:
#             message = "Processed base price not found for listing. Cannot calculate percentage differences."
#             raise HTTPException(status_code=400, detail=message)
#         try:
#             processed_pricing_details = json.loads(processed_pricing_str)
#             if not isinstance(processed_pricing_details,
#                               dict) or "base_price" not in processed_pricing_details or processed_pricing_details.get(
#                     "base_price") is None:
#                 message = "Valid processed base price not found in listing data."
#                 raise HTTPException(status_code=400, detail=message)
#             processed_base_price = float(processed_pricing_details["base_price"])
#             if processed_base_price <= 0:
#                 message = "Processed base price must be positive."
#                 raise HTTPException(status_code=400, detail=message)
#         except (json.JSONDecodeError, ValueError, TypeError) as e:
#             message = f"Error parsing processed base price: {e}"
#             raise HTTPException(status_code=500, detail=message)
#
#         # 2. Fetch and process market data for the date range
#         cursor.execute(f"SHOW TABLES LIKE '{market_table_for_range}'")
#         if not cursor.fetchone():
#             message = f"Market data table '{market_table_for_range}' not found. Cannot calculate market-specific adjustments."
#             # Still save empty adjustments if needed or just return error? For now, let's return an error.
#             # If we wanted to save "no data" as 0% adjustments, logic would change.
#             # For now, this is a critical failure for *this* operation.
#             return SpecificDatePricingResponse(
#                 listing_id=listing_id, location_name=location_name,
#                 requested_date_range={"start": start_date_str, "end": end_date_str},
#                 market_data_source_table=market_table_for_range,
#                 processed_base_price=processed_base_price,
#                 daily_percentage_adjustments={}, message=message
#             )
#
#         market_df_for_range = pd.read_sql_query(f"SELECT * FROM `{market_table_for_range}`", conn)
#         if market_df_for_range.empty:
#             message = f"Market data table '{market_table_for_range}' is empty."
#             # Similar to above, treat as failure for this op.
#             return SpecificDatePricingResponse(
#                 listing_id=listing_id, location_name=location_name,
#                 requested_date_range={"start": start_date_str, "end": end_date_str},
#                 market_data_source_table=market_table_for_range,
#                 processed_base_price=processed_base_price,
#                 daily_percentage_adjustments={}, message=message
#             )
#
#         filtered_market_df = market_df_for_range  # Default if no lat/long
#         if pd.notna(user_lat) and pd.notna(user_long):
#             try:
#                 filtered_market_df = process_dataframe(market_df_for_range, lat=user_lat, long=user_long)
#             except Exception as filter_err:
#                 print(f"API: Market filter error for specific range: {filter_err}")
#                 # Potentially proceed with unfiltered data or error out
#                 message = f"Error filtering market data: {filter_err}. Proceeding with unfiltered."
#
#         # 3. Calculate P_market_range
#         try:
#             # Ensure get_recommended_price can handle potentially empty filtered_market_df
#             if filtered_market_df.empty:
#                 market_range_base_price_raw = None  # Or some default if filtered_market_df is empty
#             else:
#                 market_range_base_price_raw = get_recommended_price(filtered_market_df, user_listing_df)
#
#             if market_range_base_price_raw is None or pd.isna(market_range_base_price_raw):
#                 message = "Could not determine market base price for the specified range. Adjustments set to 0%."
#                 market_range_base_price = processed_base_price  # Assume no change if market data insufficient
#             else:
#                 market_range_base_price = float(market_range_base_price_raw)
#         except (ValueError, TypeError) as e:
#             message = f"Error converting market range base price: {e}. Adjustments set to 0%."
#             market_range_base_price = processed_base_price  # Default to no change
#
#         # 4. Calculate base percentage difference
#         if processed_base_price > 0:  # Already checked above, but good for clarity
#             base_percentage_diff = ((market_range_base_price - processed_base_price) / processed_base_price) * 100
#         else:  # Should not happen due to earlier checks
#             base_percentage_diff = 0.0
#             message += " Processed base price was zero, diff set to 0."
#
#         # 5. Calculate daily percentage adjustments with random variation
#         current_date = start_date_obj
#         while current_date <= end_date_obj:
#             date_str = current_date.strftime("%Y-%m-%d")
#             # apply_percentage_variation expects base_percentage, returns base_percentage + random variation
#             daily_percentage_adjustments[date_str] = apply_percentage_variation(base_percentage_diff)
#             current_date += timedelta(days=1)
#
#         message = "Daily percentage adjustments calculated."
#
#         # 6. Fetch existing specific_date_pricing_json, merge, and update DB
#         existing_specific_adjustments_str = user_data.get('specific_date_pricing_json')
#         all_specific_adjustments = {}
#         if existing_specific_adjustments_str:
#             try:
#                 all_specific_adjustments = json.loads(existing_specific_adjustments_str)
#                 if not isinstance(all_specific_adjustments, dict):
#                     all_specific_adjustments = {}  # Reset if malformed
#             except json.JSONDecodeError:
#                 all_specific_adjustments = {}  # Reset if malformed
#
#         all_specific_adjustments.update(daily_percentage_adjustments)  # Merge new/overwrite existing for the range
#
#         updated_specific_adjustments_json = json.dumps(all_specific_adjustments)
#         cursor.execute(
#             f"UPDATE `{USER_LISTING_TABLE_NAME}` SET specific_date_pricing_json = %s WHERE listId = %s",
#             (updated_specific_adjustments_json, listing_id)
#         )
#         conn.commit()
#         message += " Specific date pricing adjustments saved to database."
#
#         return SpecificDatePricingResponse(
#             listing_id=listing_id,
#             location_name=location_name,
#             requested_date_range={"start": start_date_str, "end": end_date_str},
#             market_data_source_table=market_table_for_range,
#             processed_base_price=round(processed_base_price, 2) if processed_base_price is not None else None,
#             market_range_base_price=round(market_range_base_price, 2) if market_range_base_price is not None else None,
#             base_percentage_difference=round(base_percentage_diff, 2) if base_percentage_diff is not None else None,
#             daily_percentage_adjustments=daily_percentage_adjustments,
#             message=message
#         )
#
#     except HTTPException as http_exc:
#         if conn and conn.is_connected(): conn.rollback()
#         raise http_exc
#     except MySQLError as e:
#         if conn and conn.is_connected(): conn.rollback()
#         print(f"API: MySQLError in market-specific-date-range-pricing: {e}");
#         traceback.print_exc()
#         raise HTTPException(status_code=500, detail=f"Database error: {e}")
#     except Exception as e:
#         if conn and conn.is_connected(): conn.rollback()
#         print(f"API: Unexpected error in market-specific-date-range-pricing: {e}");
#         traceback.print_exc()
#         raise HTTPException(status_code=500, detail=f"Unexpected internal error: {e}")
#     finally:
#         if cursor: cursor.close()
#         if conn and conn.is_connected(): conn.close()


@app.post("/api/market-specific-date-range-pricing", summary="Calculate and Store Date-Specific Percentage Adjustments",
          response_model=SpecificDatePricingResponse)
async def market_specific_date_range_pricing_endpoint(payload: SpecificDatePricingRequest = Body(...)):
    conn, cursor = None, None
    listing_id = payload.listing_id
    location_name = payload.location_name
    start_date_str = payload.start_date
    end_date_str = payload.end_date

    try:
        start_date_obj = datetime.strptime(start_date_str, "%Y-%m-%d").date()
        end_date_obj = datetime.strptime(end_date_str, "%Y-%m-%d").date()
        if start_date_obj > end_date_obj: raise ValueError("Start date cannot be after end date.")
    except ValueError as e:
        raise HTTPException(status_code=400, detail=f"Invalid date format or range: {e}. Use YYYY-MM-DD.")

    market_table_for_range = get_date_range_market_table_name(location_name, start_date_obj, end_date_obj)

    market_range_base_price: Optional[float] = None  # This is P_market for the specific date range
    base_percentage_diff: Optional[float] = None
    message = "Processing started."

    try:
        conn = get_central_db_connection()
        cursor = conn.cursor(dictionary=True)

        # 1. Fetch user listing data
        cursor.execute(
            f"SELECT listId, lat, `long`, bedrooms, processed_pricing_json, active_pricing_details, specific_date_pricing_json FROM `{USER_LISTING_TABLE_NAME}` WHERE listId = %s",
            (listing_id,)
        )
        user_data = cursor.fetchone()
        if not user_data:
            raise HTTPException(status_code=404, detail=f"User listing '{listing_id}' not found.")

        user_listing_df = pd.DataFrame([user_data])
        user_lat, user_long = user_data.get('lat'), user_data.get('long')

        # --- Determine the reference base price (P_reference) ---
        reference_base_price: Optional[float] = None
        source_of_reference_price = "unknown"

        active_details_str = user_data.get('active_pricing_details')
        if active_details_str:
            try:
                active_details = json.loads(active_details_str)
                if isinstance(active_details, dict) and active_details.get('base_price') is not None:
                    potential_ref_base = float(active_details['base_price'])
                    if potential_ref_base > 0:
                        reference_base_price = potential_ref_base
                        source_of_reference_price = "active_pricing_details"
            except (json.JSONDecodeError, ValueError, TypeError):
                pass  # Fallback

        if reference_base_price is None:  # Fallback to processed_pricing_json
            processed_pricing_str = user_data.get('processed_pricing_json')
            if processed_pricing_str:
                try:
                    processed_pricing_details = json.loads(processed_pricing_str)
                    if isinstance(processed_pricing_details, dict) and \
                            processed_pricing_details.get("base_price") is not None and \
                            not processed_pricing_details.get("error"):  # Ensure no error in processed
                        potential_ref_base = float(processed_pricing_details["base_price"])
                        if potential_ref_base > 0:
                            reference_base_price = potential_ref_base
                            source_of_reference_price = "processed_pricing_json"
                except (json.JSONDecodeError, ValueError, TypeError):
                    pass  # Error parsing processed

        if reference_base_price is None:
            message = "Could not determine a valid reference base price from active or processed details for listing. Cannot calculate percentage differences."
            raise HTTPException(status_code=400, detail=message)

        # This will be used in the response model's "processed_base_price" field, representing the reference.
        processed_base_price_for_response = reference_base_price

        # 2. Fetch and process market data for the date range
        cursor.execute(f"SHOW TABLES LIKE '{market_table_for_range}'")
        if not cursor.fetchone():
            message = f"Market data table '{market_table_for_range}' not found. Cannot calculate market-specific adjustments."
            return SpecificDatePricingResponse(
                listing_id=listing_id, location_name=location_name,
                requested_date_range={"start": start_date_str, "end": end_date_str},
                market_data_source_table=market_table_for_range,
                processed_base_price=round(processed_base_price_for_response, 2),
                daily_percentage_adjustments={}, message=message
            )

        market_df_for_range = pd.read_sql_query(f"SELECT * FROM `{market_table_for_range}`", conn)
        if market_df_for_range.empty:
            message = f"Market data table '{market_table_for_range}' is empty."
            return SpecificDatePricingResponse(
                listing_id=listing_id, location_name=location_name,
                requested_date_range={"start": start_date_str, "end": end_date_str},
                market_data_source_table=market_table_for_range,
                processed_base_price=round(processed_base_price_for_response, 2),
                daily_percentage_adjustments={}, message=message
            )

        filtered_market_df = market_df_for_range
        if pd.notna(user_lat) and pd.notna(user_long):
            try:
                filtered_market_df = select_comparables_for_pricing(market_df_for_range, user_listing_df)
            except Exception as filter_err:
                print(f"API: Market filter error for specific range: {filter_err}")
                message = f"Error filtering market data: {filter_err}. Proceeding with unfiltered."

        # 3. Calculate P_market_range
        try:
            if filtered_market_df.empty:
                market_range_base_price_raw = None
            else:
                # get_recommended_price expects the listing_df to have amenities etc.
                # For simplicity, we'll assume user_listing_df (fetched earlier) is sufficient.
                market_range_base_price_raw = get_recommended_price(filtered_market_df, user_listing_df)

            if market_range_base_price_raw is None or pd.isna(market_range_base_price_raw):
                message = f"Could not determine market base price (P_market_range) for the specified range from '{market_table_for_range}'. Adjustments will be based on reference price only (0% diff)."
                market_range_base_price = reference_base_price  # Assume no change if market data insufficient
            else:
                market_range_base_price = float(market_range_base_price_raw)
                if market_range_base_price <= 0:
                    message = f"Market base price (P_market_range) from '{market_table_for_range}' is not positive ({market_range_base_price}). Adjustments will be based on reference price only (0% diff)."
                    market_range_base_price = reference_base_price
        except (ValueError, TypeError) as e:
            message = f"Error converting market range base price: {e}. Adjustments will be based on reference price only (0% diff)."
            market_range_base_price = reference_base_price

        # 4. Calculate base percentage difference
        base_percentage_diff = 0.0
        if reference_base_price > 0:  # Should always be true due to checks above
            base_percentage_diff = ((market_range_base_price - reference_base_price) / reference_base_price) * 100
        else:
            message += " Critical: Reference base price was zero or invalid, diff set to 0."
            # This case should have been caught earlier.

        # 5. Calculate daily percentage adjustments and prepare for new DB structure
        daily_adjustments_for_response: Dict[str, float] = {}
        new_specific_date_entries: Dict[str, Dict[str, float]] = {}

        current_date = start_date_obj
        while current_date <= end_date_obj:
            date_str = current_date.strftime("%Y-%m-%d")
            daily_adj_perc = apply_percentage_variation(base_percentage_diff)

            daily_adjustments_for_response[date_str] = round(daily_adj_perc, 2)
            new_specific_date_entries[date_str] = {
                "adj_perc": round(daily_adj_perc, 2),
                "mkt_price": round(market_range_base_price, 2)  # Store the P_market_range for this period
            }
            current_date += timedelta(days=1)

        message = f"Daily percentage adjustments calculated using reference price from '{source_of_reference_price}'."

        # 6. Fetch existing specific_date_pricing_json, merge, and update DB
        existing_specific_adjustments_str = user_data.get('specific_date_pricing_json')
        all_specific_adjustments: Dict[str, Dict[str, float]] = {}
        if existing_specific_adjustments_str:
            try:
                loaded_adjustments = json.loads(existing_specific_adjustments_str)
                if isinstance(loaded_adjustments, dict):
                    # Filter to keep only valid new-format entries
                    all_specific_adjustments = {
                        k: v for k, v in loaded_adjustments.items()
                        if isinstance(v, dict) and "adj_perc" in v and "mkt_price" in v
                    }
            except json.JSONDecodeError:
                all_specific_adjustments = {}  # Reset if malformed

        all_specific_adjustments.update(new_specific_date_entries)

        updated_specific_adjustments_json = json.dumps(all_specific_adjustments)
        cursor.execute(
            f"UPDATE `{USER_LISTING_TABLE_NAME}` SET specific_date_pricing_json = %s WHERE listId = %s",
            (updated_specific_adjustments_json, listing_id)
        )
        conn.commit()
        message += " Specific date pricing adjustments saved to database."

        return SpecificDatePricingResponse(
            listing_id=listing_id,
            location_name=location_name,
            requested_date_range={"start": start_date_str, "end": end_date_str},
            market_data_source_table=market_table_for_range,
            processed_base_price=round(processed_base_price_for_response, 2),  # This is P_reference
            market_range_base_price=round(market_range_base_price, 2),
            base_percentage_difference=round(base_percentage_diff, 2),
            daily_percentage_adjustments=daily_adjustments_for_response,
            message=message
        )

    except HTTPException as http_exc:
        if conn and conn.is_connected(): conn.rollback()
        raise http_exc
    except MySQLError as e:
        if conn and conn.is_connected(): conn.rollback()
        print(f"API: MySQLError in market-specific-date-range-pricing: {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Database error: {e}")
    except Exception as e:
        if conn and conn.is_connected(): conn.rollback()
        print(f"API: Unexpected error in market-specific-date-range-pricing: {e}");
        traceback.print_exc()
        raise HTTPException(status_code=500, detail=f"Unexpected internal error: {e}")
    finally:
        if cursor: cursor.close()
        if conn and conn.is_connected(): conn.close()


# @app.get("/api/specific-date-pricing", summary="Get All Stored Specific Date Percentage Adjustments",
#          response_model=AllSpecificDatePricingResponse)
# async def get_all_specific_date_pricing_endpoint(
#         listing_id: str = Query(..., description="Listing ID"),
#         location_name: str = Query(..., description="Location Name (for context)")
# ):
#     """Retrieves all stored date-specific percentage adjustments for the listing."""
#     data = await get_processed_data_from_db(listing_id, location_name, "specific_date_pricing_json",
#                                             "Specific date pricing adjustments")
#     # get_processed_data_from_db returns {} if column is NULL or empty JSON, which is fine.
#     return AllSpecificDatePricingResponse(
#         listing_id=listing_id,
#         location_name=location_name,
#         all_specific_date_adjustments=data if isinstance(data, dict) else {}
#     )

@app.get("/api/specific-date-pricing", summary="Get All Stored Specific Date Percentage Adjustments",
         response_model=AllSpecificDatePricingResponse)
async def get_all_specific_date_pricing_endpoint(
        listing_id: str = Query(..., description="Listing ID"),
        location_name: str = Query(..., description="Location Name (for context)")
):
    """Retrieves all stored date-specific percentage adjustments for the listing."""
    # get_processed_data_from_db will return the raw JSON content (now a dict of dicts)
    # or an empty dict if column is NULL/empty or an error dict.
    raw_data_from_db = await get_processed_data_from_db(
        listing_id,
        location_name,
        "specific_date_pricing_json",
        "Specific date pricing adjustments"
    )

    adj_percentages_for_response: Dict[str, float] = {}

    if isinstance(raw_data_from_db, dict):
        # Check if it's not an error structure like {"error": "..."}
        if "error" not in raw_data_from_db:
            for date_str, entry_data in raw_data_from_db.items():
                if isinstance(entry_data, dict) and "adj_perc" in entry_data:
                    try:
                        adj_percentages_for_response[date_str] = float(entry_data["adj_perc"])
                    except (ValueError, TypeError):
                        print(f"GET_SPECIFIC_PRICING ({listing_id}): Invalid adj_perc for {date_str}. Skipping.")
                elif isinstance(entry_data, (int, float)):  # Handle potential old format data
                    adj_percentages_for_response[date_str] = float(entry_data)
                    print(f"GET_SPECIFIC_PRICING ({listing_id}): Old format data found for {date_str}.")
                # else: Malformed entry, skip or log
        # else: raw_data_from_db is an error dict, adj_percentages_for_response remains empty

    return AllSpecificDatePricingResponse(
        listing_id=listing_id,
        location_name=location_name,
        all_specific_date_adjustments=adj_percentages_for_response
    )

@app.get("/", summary="Root Endpoint", include_in_schema=False)
async def root(): return {"status": "Dynamic Pricing API is running", "database": CENTRAL_DB_NAME,
                          "user_table": USER_LISTING_TABLE_NAME}


if __name__ == "__main__":
    import uvicorn

    print(f"--- Starting Dynamic Pricing API v{app.version} ---")
    uvicorn.run(app, host="training.paperbirdtech.com", port=8080)
